# coding=utf-8

from sandbox.projects.common import binary_task
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.java import metrika_java_test_stand_create
from sandbox.projects.metrika.java.metrika_java_development import development_helper, development_info
from sandbox.projects.metrika.utils.base_metrika_task import with_parents
from sandbox.projects.metrika.utils.parameters import hide, TrackerIssue, DataCenterParameters
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.sdk2 import parameters

REPORT_TTL = 30


@with_parents
class MetrikaJavaDevelopment(pipeline.PipelineBaseTask):
    """
    Сборка и тестирование java-демонов
    """

    @property
    def development(self):
        return development_info.DevelopmentInfo(self.Context.development_info_state)

    class Context(pipeline.PipelineBaseTask.Context):
        development_info_state = development_info.DevelopmentInfo().state
        daemon_list = []
        ref_version = None
        build_packages_task_ids = []
        deploy_test_stands_task_ids = []
        run_tests_task_ids = []

    class Parameters(utils.CommonParameters):
        description = "Сборка и тестирование java-демонов Метрики"

        tracker_issue = TrackerIssue("Ключ тикета", description="Ключ тикета в Трекере.")

        arcadia_url = parameters.ArcadiaUrl(
            "URL Аркадии", required=False, default="",
            description="Ветка или коммит, из которой будут собраны демона и запущены тесты. Если не задана, используется ключ тикета. Если и он не указан - arcadia-arc:/#trunk."
        )

        with parameters.Group("Настройки этапа сборки") as build_packages_group:
            build_packages = parameters.Bool(
                "Сборка", required=True, default=True,
                description="Если демона не будут собираться, можно указать уже собранные версии пакетов."
            )

            with build_packages.value[True]:
                daemon_list = parameters.List(
                    "Список демонов", required=True,
                    description="Java-демона Метрики, сборка и тестирование которых будет осуществлено.",
                )

            with build_packages.value[False]:
                daemon_dictionary = parameters.Dict(
                    "Список демонов", required=True, default={},
                    description="Java-демона Метрики, тестирование которых будет осуществлено."
                )

        with parameters.Group("Настройки этапа создания тестовых стендов") as create_test_stands_group:
            create_test_stands = parameters.Bool(
                "Создание стандартных тестовых стендов", required=True, default=True,
                description="По умолчанию имя стенда формируется из автора и ключа тикета."
            )

            with create_test_stands.value[True]:
                test_stand_name = parameters.String("Имя тестового стенда", required=True, default="{author}-{tracker_issue}")

                vanilla = parameters.Bool("Нефильтрованный инстанс", required=True, default=False, description="Использовать ли инстанс vanilla.")

                sql_script = parameters.String(
                    "SQL скрипт", required=False, default="", multiline=True,
                    description="Скрипт для применения к mysql базе."
                )

                create_frontend_stand = parameters.Bool("Создать стенд фронтенда", default=False)

                with create_frontend_stand.value[True]:
                    frontend_arcadia_url = parameters.ArcadiaUrl(
                        "URL Аркадии фронтенда", required=True, default="arcadia-arc:/#trunk",
                        description="Ветка или коммит, из которой будет собран фронтенд"
                    )

            with create_test_stands.value[False]:
                custom_test_stand_name = parameters.String(
                    "Имя произвольного тестового стенда", required=True, default="",
                    description="Стенд для запуска тестов, созданный ранее."
                )

            data_center_params = DataCenterParameters()

        with parameters.Group("Настройки этапа тестирования") as run_tests_group:
            run_tests = parameters.Bool(
                "Тестирование", required=True, default=True,
                description="Можно не запускать тесты."
            )

            custom_tests_run = parameters.Bool(
                "Произвольная конфигурация тестирования", required=False, default=False,
                description="Позволяет запускать только определенные тесты."
            )

            with custom_tests_run.value[True]:
                custom_tests_run_configuration = parameters.String(
                    "Конфигурация тестирования", required=False, default=development_info.TESTS_RUN_CONFIGURATION_TEMPLATE, multiline=True,
                    description="Можно задавать шаблон. Если пустое - будет использована конфигурация по умолчанию."
                )

        with parameters.Group("Настройки этапа отправки уведомлений") as send_notifications_group:
            send_notifications = parameters.Bool(
                "Уведомления", required=True, default=True,
                description="Можно не отправлять уведомлений."
            )

            with send_notifications.value[True]:
                send_telegram_notifications = parameters.Bool(
                    "Уведомления в Telegram", required=True, default=True,
                    description="Можно не отправлять уведомлений в Telegram."
                )

                with send_telegram_notifications.value[True]:
                    telegram_notifications_recipients = parameters.List(
                        "Получатели Telegram уведомлений", required=False, default=[],
                        description="По умолчанию уведомление отправляется автору таски или в общий канал."
                    )

                    send_telegram_notifications_on_success = parameters.Bool(
                        "Уведомления об успехе", required=False, default=False,
                        description="Отправлять ли Telegram уведомления об успешнно пройденных тестах."
                    )

                send_email_notifications = parameters.Bool(
                    "Уведомления по e-mail", required=True, default=False,
                    description="Можно не отправлять уведомлений по e-mail."
                )

                with send_email_notifications.value[True]:
                    email_notifications_recipients = parameters.List(
                        "Получатели e-mail уведомлений", required=False, default=[],
                        description="Могут быть указаны как люди, так и рассылки. "
                                    "По умолчанию уведомление отправляется автору."
                    )

                    send_email_notifications_on_success = parameters.Bool(
                        "Уведомления об успехе", required=False, default=False,
                        description="Отправлять ли e-mail уведомления об успешнно пройденных тестах."
                    )

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    def on_save(self):
        development_helper.DevelopmentHelper.check_daemons(self, self.Parameters.daemon_list if self.Parameters.build_packages else self.Parameters.daemon_dictionary)

    def on_enqueue(self):
        self.development.author = self.real_author
        self.development.arcadia_url = self.Parameters.arcadia_url or (
            "arcadia-arc:/#users/{}/{}".format(self.development.author, self.Parameters.tracker_issue) if self.Parameters.tracker_issue else "arcadia-arc:/#trunk"
        )

        self.Context.daemon_list = self.Parameters.daemon_list if self.Parameters.build_packages else self.Parameters.daemon_dictionary.keys()

        development_helper.DevelopmentHelper.update_description(self, self.development.author, self.Context.daemon_list, self.development.arcadia_url)

    def create_stages(self):
        return [
            (self.stage_build_packages, "Сборка"),
            (self.stage_create_test_stands, "Тестовые стенды"),
            (self.stage_run_tests, "Тесты"),
            (self.stage_send_notifications, "Уведомления")
        ]

    def stage_build_packages(self):
        if self.Parameters.tracker_issue:
            development_helper.DevelopmentHelper.add_issue_tags(self, self.Context.daemon_list)

        if self.Parameters.build_packages:
            self.Context.all_programs = development_helper.DevelopmentHelper.get_all_programs(self.development.arcadia_url, "metrika/java")

            version = "2.42.{}".format(self.id)
            daemon_dictionary = dict((daemon, version) for daemon in self.Parameters.daemon_list)
            self.development.fill_daemons_info_state(daemon_dictionary)

            build_tasks = development_helper.DevelopmentHelper.get_build_tasks(
                self, self.development.author, self.Context.all_programs,
                self.Parameters.daemon_list, version, self.development.arcadia_url
            )
            if self.development.has_b2b_stands() and self.development.arcadia_url != "arcadia-arc:/#trunk":
                self.development.ref_version = version + "-ref"
                build_tasks.extend(development_helper.DevelopmentHelper.get_build_tasks(
                    self, self.development.author, self.Context.all_programs,
                    self.Parameters.daemon_list, self.development.ref_version, development_helper.DevelopmentHelper.get_branch_point(self.development.arcadia_url))
                )

            self.run_subtasks(build_tasks, subtasks_variable=self.Context.build_packages_task_ids, info=False)

            development_helper.DevelopmentHelper.set_build_info(self, self.Context.build_packages_task_ids)
        else:
            daemon_dictionary = dict((daemon, version or development_helper.DevelopmentHelper.get_stable_version(daemon)) for daemon, version in self.Parameters.daemon_dictionary.items())
            self.development.fill_daemons_info_state(daemon_dictionary)

            development_helper.DevelopmentHelper.set_build_packages_info(self, daemon_dictionary)

    def stage_create_test_stands(self):
        with self.memoize_stage.create_test_stands(commit_on_entrance=False):
            self.development.fill_deploy_stands_names(
                self.Parameters.test_stand_name or self.Parameters.custom_test_stand_name,
                self.Parameters.tracker_issue, self.development.ref_version,
                self.Parameters.vanilla
            )

        if self.Parameters.create_test_stands:
            deploy_test_stands_tasks = []
            for deploy_test_stand in self.development.daemons_deploy_stands:
                deploy_test_stand_parameters = {
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.name: deploy_test_stand.name,
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.daemon_name: deploy_test_stand.daemon_name,
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.version: deploy_test_stand.version,
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.bishop_environment_prefix: deploy_test_stand.bishop_environment_prefix,
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.sql_script: self.Parameters.sql_script,
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.data_center: self.Parameters.data_center,
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.create_frontend_stand: self.Parameters.create_frontend_stand,
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.maas_parent.name: deploy_test_stand.maas_parent,
                    metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.frontend_arcadia_url: self.Parameters.arcadia_url,
                }
                deploy_test_stands_tasks.append((metrika_java_test_stand_create.MetrikaJavaTestStandCreate, deploy_test_stand_parameters))
            self.run_subtasks(deploy_test_stands_tasks, subtasks_variable=self.Context.deploy_test_stands_task_ids, info=False)

        development_helper.DevelopmentHelper.set_create_deploy_test_stands_info(self, self.development.daemons_deploy_stands)

    def stage_run_tests(self):
        if self.Parameters.run_tests:
            for test_stand in self.development.daemons_deploy_stands:
                test_stand.fqdn = metrika_java_test_stand_create.MetrikaJavaTestStandCreate.get_fqdn(test_stand.name, test_stand.daemon_name, self.Parameters.data_center)

            if self.Parameters.custom_tests_run and self.Parameters.custom_tests_run_configuration:
                development_helper.DevelopmentHelper.apply_custom_tests_run_configuration(
                    self.development.daemons_info,
                    self.Parameters.custom_tests_run_configuration
                )

            self.run_subtasks(development_helper.DevelopmentHelper.get_tests_tasks(
                self.development.daemons_info,
                self.development.arcadia_url,
                self.Context.all_programs,
                REPORT_TTL,
                self.Parameters.tracker_issue
            ), subtasks_variable=self.Context.run_tests_task_ids, after_subtasks_enqueued=self.after_tests_enqueued, info=False)
        else:
            self.set_info("Запуска тестов не будет.")

    def after_tests_enqueued(self):
        daemons_under_auto_tests_info = [info for info in self.development.daemons_info if info.under_auto_tests]
        for info, task_id in zip(daemons_under_auto_tests_info, self.Context.run_tests_task_ids):
            info.tests_task_id = task_id
        development_helper.DevelopmentHelper.set_run_tests_info(self, self.development.daemons_info)

    def stage_send_notifications(self):
        all_tests_passed = development_helper.DevelopmentHelper.all_tests_passed(self.development.daemons_info)

        send_telegram_notifications = self.Parameters.send_telegram_notifications and (self.Parameters.send_telegram_notifications_on_success or not all_tests_passed)

        send_email_notifications = self.Parameters.send_email_notifications and (self.Parameters.send_email_notifications_on_success or not all_tests_passed)

        send_notifications = self.Parameters.send_notifications and (send_telegram_notifications or send_email_notifications)

        if send_notifications:
            if send_telegram_notifications:
                development_helper.DevelopmentHelper.send_telegram_notifications(self, self.Parameters.telegram_notifications_recipients, self.development.daemons_info)

            if send_email_notifications:
                development_helper.DevelopmentHelper.send_email_notifications(self, self.Parameters.email_notifications_recipients, self.development.daemons_info)

            development_helper.DevelopmentHelper.set_send_notifications_info(self, send_telegram_notifications, send_email_notifications)
        else:
            self.set_info("Отправки уведомлений не будет.")
