# coding=utf-8
import logging

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.utils import get_task_link
from sandbox.projects.common import link_builder
from sandbox.projects.common.vcs import arc, util
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.java.utils import metrika_java_helper
from sandbox.projects.metrika.utils import notifications

logger = logging.getLogger("development-helper")


class DevelopmentHelper(metrika_java_helper.MetrikaJavaHelper):

    @staticmethod
    def add_issue_tags(task, daemon_list):
        try:
            issue = task.st_client.issues[task.Parameters.tracker_issue]
            issue.update(tags=list(set(issue.tags + daemon_list)))
        except Exception as exception:
            task.set_info(exception.message)
            logger.warning("Ошибка установки тегов в тикет", exc_info=True)

    @staticmethod
    def get_branch_point(arcadia_url):
        clone_path, commit_id, vcs_type = util.extract_path_details(arcadia_url)
        logger.info("Arcadia url '{url}' parsed into clone_path='{clone_path}', commit_id='{commit_id}', vcs_type='{vcs_type}'"
                    .format(url=arcadia_url, clone_path=clone_path, commit_id=commit_id, vcs_type=vcs_type))
        if vcs_type == util.ARC_VCS:
            with arc.Arc().mount_path(clone_path, commit_id, fetch_all=False) as mount_point:
                merge_base = arc.Arc().get_merge_base(mount_point, branch=commit_id)
                logger.info("Arc merge-base: {}".format(merge_base))
                return DevelopmentHelper.get_arcadia_url(merge_base)
        else:
            raise errors.TaskFailure("Unsupported Arcadia URL for finding branch point: {}".format(arcadia_url))

    @staticmethod
    def apply_custom_tests_run_configuration(daemons_info, configuration):
        for info in daemons_info:
            info.tests_run_configuration_template = configuration

    @staticmethod
    def set_run_tests_info(task, daemons_info):
        daemons_info_with_tests_task_id = [info for info in daemons_info if info.tests_task_id]

        if daemons_info_with_tests_task_id:
            template = "{} - <a href=\"{}\">{}</a>"

            run_tests_info = "Тесты:<br/>" + "<br/>".join(
                template.format(info.name, get_task_link(info.tests_task_id), info.tests_task_id) for info in
                daemons_info_with_tests_task_id)

            task.set_info(run_tests_info, do_escape=False)

    @staticmethod
    def all_tests_passed(daemons_info):
        return all(sdk2.Task[info.tests_task_id].is_all_tests_passed for info in daemons_info if info.tests_task_id)

    @staticmethod
    def send_telegram_notifications(task, recipients, daemons_info):
        recipients = recipients if any(recipient for recipient in recipients) else task.real_author

        test_links = dict(
            (info.name,
             link_builder.HREF_TO_ITEM.format(link=get_task_link(info.tests_task_id), name="#" + str(info.tests_task_id)) +
             " <b>{}</b>".format("тесты пройдены" if sdk2.Task[info.tests_task_id].is_all_tests_passed else "тесты не пройдены")
             if info.tests_task_id else "не тестировался") for info in daemons_info
        )

        telegram_message = "\n".join("<b>{}</b> {}".format(name, utils.encode(link)) for name, link in test_links.items())

        notifications.Notifications.telegram(task, recipients, telegram_message)

    @staticmethod
    def send_email_notifications(task, recipients, daemons_info):
        recipients = recipients if any(recipient for recipient in recipients) else task.real_author

        test_results = dict((info.name, sdk2.Task[info.tests_task_id].get_html_comment() if info.tests_task_id else "не тестировался") for info in daemons_info)

        email_subject = "{} {}".format(task.type.name, DevelopmentHelper._get_daemon_list(daemons_info))

        email_body = "<br/>".join("<b>{}</b> {}<br/><hr/>".format(name, utils.encode(report)) for name, report in test_results.items())

        notifications.Notifications.email(task, recipients, email_subject, email_body)

    @staticmethod
    def set_send_notifications_info(task, telegram, email):
        send_notifications_info = "Уведомления:"
        if telegram:
            send_notifications_info += "<br/>telegram"
        if email:
            send_notifications_info += "<br/>e-mail"

        task.set_info(send_notifications_info, do_escape=False)
