# coding=utf-8
import contextlib
import logging

from datetime import timedelta

from sandbox import sdk2
from sandbox.projects.common import binary_task
from sandbox.projects.common.decorators import memoized_property
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.utils import base_metrika_task, maas, settings
from sandbox.projects.metrika.utils.maas import MaasProvider
from sandbox.projects.metrika.utils.parameters import DataCenterParameters
from sandbox.projects.metrika.utils.mixins import juggler_reporter
from sandbox.sdk2 import parameters


@base_metrika_task.with_parents
class MetrikaJavaMaasParentPrepare(base_metrika_task.BaseMetrikaTask, juggler_reporter.JugglerReporterMixin, sdk2.Task):
    """
    Создание родительских инстансов MaaS для java-демонов Метрики
    """

    class Parameters(utils.CommonParameters):
        description = "Создание родительских инстансов MaaS для java-демонов Метрики"

        parent_id = sdk2.parameters.Integer("Идентификатор родительского инстанса", required=True)
        yav_token = sdk2.parameters.YavSecretWithKey('Vault token', required=True, default='{}#vault_oauth_token'.format(settings.rma_yav_uuid),
                                                     description="Секрет с токеном для доступа к секретнице")
        bishop_token = sdk2.parameters.YavSecretWithKey('Bishop token', required=True, default='{}#bishop_oauth_token'.format(settings.rma_yav_uuid),
                                                        description="Секрет с токеном для доступа к bishop")

        data_center_params = DataCenterParameters()

        sql_script = parameters.String(
            "SQL скрипт", required=False, default="", multiline=True,
            description="Скрипт для применения к базе."
        )

        _binary = binary_task.binary_release_parameters_list(stable=True)

    def _juggler_predicate(self, status):
        return self.author == settings.login

    @memoized_property
    def vault_client(self):
        import metrika.pylib.vault as vault
        return vault.VaultClient(auth_type="oauth", oauth_token=self.Parameters.yav_token.value())

    @memoized_property
    def maas_provider(self):
        return MaasProvider(self.Parameters.bishop_token.value(), self.Parameters.data_center)

    def get_vanilla_scripts(self):
        return {
            "conv_main": utils.read_sources_file("vanilla/conv_main.sql")
        }

    def get_distilled_scripts(self):
        return {
            "conv_main": utils.read_sources_file("distilled/conv_main.sql")
        }

    def get_parent_scripts(self):
        return {
            "conv_main": utils.read_sources_file("vanilla/conv_main.sql"),
            "audience_main": utils.read_sources_file("vanilla/audience_main.sql"),
            "audience_rbac": utils.read_sources_file("vanilla/audience_rbac.sql")
        }

    def suffixize_name(self, name):
        return name + "-80"

    @property
    def vanilla_name(self):
        return self.suffixize_name("vanilla")

    @property
    def distilled_name(self):
        return self.suffixize_name("distilled")

    @property
    def parent_name(self):
        return self.suffixize_name("Parent")

    def on_execute(self):
        self.create_parent(self.Parameters.parent_id, self.parent_name, prepare_sql=self.get_parent_scripts(), users_sql=self.Parameters.sql_script)
        vanilla_id = self.create_parent(self.Parameters.parent_id, self.vanilla_name, prepare_sql=self.get_vanilla_scripts(), users_sql=self.Parameters.sql_script)
        self.create_parent(vanilla_id, self.distilled_name, prepare_sql=self.get_distilled_scripts())

    def create_parent(self, parent_id, parent_name, prepare_sql=None, users_sql=None):
        parent_ttl = 7
        parent_temporary_name = parent_name + "-proto"
        parent_params = {
            "name": parent_temporary_name,
            "version_tag": maas.VERSION_TAG,
            "memory_limit": 8,
            "ttl": int(timedelta(days=parent_ttl).total_seconds())
        }

        if parent_id:
            parent_params["parent_id"] = parent_id

        instance = self.maas_provider.get_maas_client()
        instance.create(**parent_params)

        if prepare_sql:
            self.__execute_mysql(instance, prepare_sql)

        if users_sql:
            self.__execute_mysql(instance, {"": users_sql})

        instance.update(name=parent_name)

        return instance.id

    def __execute_mysql(self, instance, db_queries):
        import pymysql.cursors

        user = 'dba'
        password = self.vault_client.get_version('sec-01f2p6kyz96gzz6j8k0zjs9851')['value'][user]

        for database, queries in db_queries.items():
            connection_args = {
                "host": instance.host,
                "port": instance.ports["mysql"],
                "user": user,
                "password": password
            }
            if database:
                connection_args["db"] = database
            with contextlib.closing(pymysql.connect(**connection_args)) as connection:
                for query in queries.split(";"):
                    if query.strip():
                        logging.info("Execute SQL query:\n%s", query)
                        connection.begin()
                        with connection.cursor() as cursor:
                            cursor.execute(query)
                        connection.commit()
