# coding=utf-8
import json

from sandbox import sdk2
from sandbox.common import errors
from sandbox.common.types import task as task_type
from sandbox.projects.common import binary_task
from sandbox.projects.common.build.ya_package_config import consts
from sandbox.projects.common.decorators import retries
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.java import metrika_java_integration_tests_run, metrika_java_test_stand_create
from sandbox.projects.metrika.java.metrika_java_release import release_helper, release_info
from sandbox.projects.metrika.utils import base_metrika_task, settings
from sandbox.projects.metrika.utils.mixins import juggler_reporter, releasable
from sandbox.projects.metrika.utils.parameters import hide, DataCenterParameters
from sandbox.projects.metrika.utils.pipeline import pipeline
from sandbox.sdk2 import parameters

REPORT_TTL = 730


@base_metrika_task.with_parents
class MetrikaJavaRelease(pipeline.PipelineBaseTask, releasable.ReleasableMixin, juggler_reporter.JugglerReporterMixin):
    """
    Готовит релиз java демонов Метрики
    """

    @property
    def release(self):
        return release_info.ReleaseInfo(self.Context.release_info_state)

    @property
    def release_template(self):
        if not self.Context.released:
            return sdk2.ReleaseTemplate(None, None, None, [task_type.ReleaseStatus.TESTING])
        else:
            return sdk2.ReleaseTemplate(None, None, None, [task_type.ReleaseStatus.STABLE, task_type.ReleaseStatus.PRESTABLE, task_type.ReleaseStatus.TESTING])

    class Context(pipeline.PipelineBaseTask.Context):
        release_info_state = release_info.ReleaseInfo().state
        build_packages_task_ids = []
        deploy_test_stands_task_ids = []
        run_tests_task_ids = []

    class Parameters(utils.CommonParameters):
        description = "Релиз java-демонов Метрики"

        daemon_list = parameters.List("Список демонов", required=True, default="",
                                      description="Java-демона Метрики, релиз которых будет осуществлен.")

        is_new = parameters.Bool("Новый релиз", required=True, default=True,
                                 description="У нового релиза увеличивается мажорная версия, у старого - минорная.")

        with is_new.value[True]:
            new_release_arcadia_url = parameters.ArcadiaUrl("URL Аркадии", required=False, default="",
                                                            description="Опционально ветка или коммит, от которого отводить новую релизную ветку. "
                                                                        "Если пустое - отводится от текущего arcadia-arc:/#trunk.")

            force_new_release = parameters.Bool("Принудительно выпустить новый релиз", required=True, default=False,
                                                description="Игнорировать проверку наличия в тестинге невыкаченного в прод релиза.")

        with is_new.value[False]:
            previous_release_arcadia_url = parameters.ArcadiaUrl("Релизный бранч", required=True, default="", description="Релизный бранч, который нужно продолжить. "
                                                                                                                          "Формат: arcadia-arc:/#releases/metrika/java/{version}")

            wait_for_tests = parameters.Bool("Ждать прохождения тестов", required=True, default=True,
                                             description="Возможность выкладки релиза появится только после завершения тестов.")

        data_center_params = DataCenterParameters()

        _binary = hide(binary_task.binary_release_parameters_list(stable=True))

    def on_save(self):
        release_helper.ReleaseHelper.check_daemons(self, self.Parameters.daemon_list)

    def on_enqueue(self):
        self.release.author = self.real_author

        release_helper.ReleaseHelper.update_description(self, self.release.author, self.Parameters.daemon_list, self.release.release_version)

    @retries(3)
    def on_prepare(self):
        if not self.Context.all_programs:
            with sdk2.helpers.ProgressMeter("Retrieve all programs"):
                if self.Parameters.is_new:
                    if self.Parameters.new_release_arcadia_url:
                        self.Context.arcadia_url = self.Parameters.new_release_arcadia_url
                    else:
                        # Если ветка или коммит не задана - выкачиваем trunk
                        self.Context.arcadia_url = release_helper.ReleaseHelper.get_arcadia_url("trunk")
                else:
                    self.Context.arcadia_url = self.Parameters.previous_release_arcadia_url

            self.Context.all_programs = release_helper.ReleaseHelper.get_all_programs(self.Context.arcadia_url, "metrika/java")

    def create_stages(self):
        return [
            (self.stage_check_previous_releases, "Проверка"),
            (self.stage_create_release_branch, "Релизный бранч"),
            (self.stage_get_changelog, "Лог изменений"),
            (self.stage_create_release_issues, "Релизные тикеты"),
            (self.stage_build_packages, "Сборка"),
            (self.stage_create_test_stands, "Тестовые стенды"),
            (self.stage_create_tests_issues, "Тикеты на анализ тестов"),
            (self.stage_run_tests, "Тесты")
        ]

    def stage_check_previous_releases(self):
        if self.Parameters.is_new and not self.Parameters.force_new_release:
            release_helper.ReleaseHelper.check_previous_releases(self.Parameters.daemon_list)

    @retries(3, delay=30, exceptions=errors.TaskError)
    def stage_create_release_branch(self):
        with sdk2.helpers.ProgressMeter("Retrieve release version"):
            if self.Parameters.is_new:
                self.release.previous_release_version = release_helper.ReleaseHelper.get_latest_release_version()
            else:
                self.release.previous_release_version = release_helper.ReleaseHelper.get_version_from_arcadia_url(self.Parameters.previous_release_arcadia_url)

        self.release.release_version = release_helper.ReleaseHelper.increase_version(self.release.previous_release_version, self.Parameters.is_new)

        self.release.fill_daemons_info_state(self.Parameters.daemon_list)

        branch = release_helper.ReleaseHelper.get_release_branch(self.release.release_version)

        release_helper.ReleaseHelper.create_release_branch(self, branch, self.Context.arcadia_url)

        release_helper.ReleaseHelper.update_description(self, self.release.author, self.Parameters.daemon_list, self.release.release_version)

        release_helper.ReleaseHelper.set_create_release_branch_info(self, branch)

        self.Context.branch_url = release_helper.ReleaseHelper.get_arcadia_url(release_helper.ReleaseHelper.get_release_branch(self.release.release_version))

    def stage_get_changelog(self):
        release_helper.ReleaseHelper.get_previous_versions(self.release, self.Parameters.is_new)

        release_helper.ReleaseHelper.get_changelogs(self.release, self.Parameters.is_new, self.st_client)

    def stage_create_release_issues(self):
        self.Context.release_issue_key = release_helper.ReleaseHelper.create_release_issues(self, self.release)[0]

        release_helper.ReleaseHelper.set_create_release_issues_info(self, self.release.daemons_info)

    def stage_build_packages(self):
        self.run_subtasks(release_helper.ReleaseHelper.get_build_tasks(
            self, self.release.author,
            self.Context.all_programs, self.Parameters.daemon_list, self.release.release_version
        ), subtasks_variable=self.Context.build_packages_task_ids, info=False)

        release_helper.ReleaseHelper.set_build_info(self, self.Context.build_packages_task_ids)

    def stage_create_test_stands(self):
        with self.memoize_stage.create_test_stands(commit_on_entrance=False):
            self.release.fill_deploy_stands_names()

        deploy_test_stands_tasks = []
        for deploy_test_stand in self.release.daemons_deploy_stands:
            deploy_test_stand_parameters = {
                metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.name.name: deploy_test_stand.name,
                metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.daemon_name.name: deploy_test_stand.daemon_name,
                metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.version.name: deploy_test_stand.version,
                metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.bishop_environment_prefix.name: deploy_test_stand.bishop_environment_prefix,
                metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.data_center.name: self.Parameters.data_center,
                metrika_java_test_stand_create.MetrikaJavaTestStandCreate.Parameters.maas_parent.name: deploy_test_stand.maas_parent
            }
            deploy_test_stands_tasks.append((metrika_java_test_stand_create.MetrikaJavaTestStandCreate, deploy_test_stand_parameters))
        self.run_subtasks(deploy_test_stands_tasks, subtasks_variable=self.Context.deploy_test_stands_task_ids, info=False)

        release_helper.ReleaseHelper.set_create_deploy_test_stands_info(self, self.release.daemons_deploy_stands)

    def stage_create_tests_issues(self):
        release_helper.ReleaseHelper.create_tests_issues(self, settings, self.release)

        release_helper.ReleaseHelper.set_create_tests_issues_info(self, self.release.daemons_info)

    def after_tests_enqueued(self):
        release_helper.ReleaseHelper.set_run_tests_info(self, self.Context.run_tests_task_ids)

    def stage_run_tests(self):
        for test_stand in self.release.daemons_deploy_stands:
            test_stand.fqdn = metrika_java_test_stand_create.MetrikaJavaTestStandCreate.get_fqdn(test_stand.name, test_stand.daemon_name)

        self.run_subtasks(
            release_helper.ReleaseHelper.get_tests_tasks(
                self.release.daemons_info,
                self.Context.branch_url,
                self.Context.all_programs,
                REPORT_TTL,
                is_release=True
            ),
            subtasks_variable=self.Context.run_tests_task_ids,
            wait=self.Parameters.wait_for_tests or self.Parameters.is_new,
            after_subtasks_enqueued=self.after_tests_enqueued,
            info=False
        )

    def on_release(self, parameters):
        if parameters['release_status'] == task_type.ReleaseStatus.TESTING:
            releases = {}
            for build_task in self.Context.build_packages_task_ids:
                build_task = sdk2.Task[build_task]
                if build_task.Context.ya_deploy_release_ids:
                    releases[build_task.Parameters.package_type] = next(r for r in build_task.Context.ya_deploy_release_ids if r.endswith(task_type.ReleaseStatus.TESTING))
            self.run_subtasks([
                (
                    metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun,
                    {
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.arcadia_url: self.Context.branch_url,
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.daemon: daemon.name,
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.version: daemon.version,
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.release: releases[consts.PackageType.DOCKER.value],
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.stage: daemon.integration_tests_group,
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.ticket: daemon.tests_issue_key,
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.need_semaphore: daemon.need_semaphore,
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.build_configuration: json.loads(daemon.integration_tests_build_configuration),
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.run_configuration: json.loads(daemon.integration_tests_run_configuration),
                        metrika_java_integration_tests_run.MetrikaJavaIntegrationTestsRun.Parameters.report_ttl: REPORT_TTL
                    }
                )
                for daemon in self.release.daemons_info
                if daemon.integration_tests_host
            ], wait=False)
        self.Context.released = True
