# coding=utf-8
import logging
import re
from distutils import version
from itertools import groupby

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.common.vcs import arc
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.java.utils import metrika_java_helper
from sandbox.projects.metrika.utils import arcanum_api, settings, vcs
from sandbox.sdk2.vcs import svn

logger = logging.getLogger("release-helper")


class ReleaseHelper(metrika_java_helper.MetrikaJavaHelper):
    release_branch_prefix = "releases/metrika/java/"

    @staticmethod
    def update_description(task, author, daemon_list, version=None):
        ReleaseHelper._update_description(task, ("{0} версии {1} от {2}" if version else "{0} от {2}").format(
            ", ".join(ReleaseHelper._green(daemon) for daemon in daemon_list),
            ReleaseHelper._grey(version),
            ReleaseHelper._blue(author)
        ))

    @staticmethod
    def check_previous_releases(daemons_list):
        for daemon in daemons_list:
            stable_version = ReleaseHelper.get_stable_package_version(daemon)
            testing_version = ReleaseHelper.get_package_version(daemon, "testing") or ReleaseHelper.get_package_version(daemon, "test")
            if testing_version and stable_version and testing_version != stable_version:
                raise errors.TaskError("Демон {} имеет невыложенный релиз (версия в тестинге {}, версия в проде {})".format(daemon, testing_version, stable_version))

    @staticmethod
    def get_latest_release_version():
        arcanum = arcanum_api.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))
        all_branches = arcanum.get_branches(ReleaseHelper.release_branch_prefix)
        return str(max(
            [version.StrictVersion(ReleaseHelper.get_version_from_branch(branch)) for branch in all_branches if branch.startswith(ReleaseHelper.release_branch_prefix)]
        ))

    @staticmethod
    def get_version_from_branch(branch):
        return branch.replace(ReleaseHelper.release_branch_prefix, "")

    @staticmethod
    def get_version_from_arcadia_url(arcadia_url):
        return ReleaseHelper.get_version_from_branch(svn.Arcadia.get_revision(arcadia_url))

    @staticmethod
    def get_release_branch(release_version):
        return ReleaseHelper.release_branch_prefix + release_version

    @staticmethod
    def get_release_arcadia_url(release_version):
        return ReleaseHelper.get_arcadia_url(ReleaseHelper.get_release_branch(release_version))

    @staticmethod
    def build_version(major, minor, patch):
        return "{}.{}.{}".format(major, minor, patch)

    @staticmethod
    def increase_version(original_version, is_new):
        major, minor, patch = version.StrictVersion(original_version).version
        if is_new:
            minor += 1
            patch = 1
        else:
            patch += 1
        return ReleaseHelper.build_version(major, minor, patch)

    @staticmethod
    def get_previous_versions(release, is_new):
        for info in release.daemons_info:
            info.previous_version = ReleaseHelper.get_stable_version(info.name, raise_for_fail=False) if is_new else release.previous_release_version

    @staticmethod
    def create_release_branch(task, branch, start_point_url):
        from metrika.pylib.zkconnect import get_zk
        zk = get_zk(zapi_group='common-proxy-bypass', zapi_env='testing')
        zk.start()
        lock_path = '/admins/release'
        release_path = lock_path + '/' + branch.split('/')[-1]
        with zk.Lock(lock_path, task.id):
            if zk.exists(release_path):
                raise errors.TaskError('Branch {} already exists'.format(branch))
            else:
                arc_client = arc.Arc()
                with vcs.mount_arc(start_point_url) as arcadia:
                    arc_client.checkout(arcadia, branch, True, svn.Arcadia.get_revision(start_point_url))
                    arc_client.push(arcadia, branch)
                zk.create(release_path)

    @staticmethod
    def get_changelogs(release, is_new, st_client):
        project_path = "metrika/java"

        arc_client = arc.Arc()
        with vcs.mount_arc(ReleaseHelper.get_arcadia_url("trunk")) as arcadia:
            for info in release.daemons_info:
                start = ReleaseHelper.get_release_branch(info.previous_version) if info.previous_version else None
                end = ReleaseHelper.get_release_branch(info.version)

                if start:
                    arc_client.fetch(arcadia, start)
                arc_client.fetch(arcadia, end)
                arc_client.checkout(arcadia, end, force=True)
                if is_new:
                    info.changelog = arc_client.log(arcadia, path=project_path, start_commit=start, end_commit=end, max_count=1000)
                else:
                    if ReleaseHelper.__get_version_patch(info.previous_version) == 1:
                        parent_branch = "trunk"
                    else:
                        parent_branch = ReleaseHelper.get_release_branch(ReleaseHelper.__decrease_version_patch(info.previous_version))
                        arc_client.fetch(arcadia, parent_branch)
                    info.changelog = arc_client.log(arcadia, path=project_path, start_commit=parent_branch, end_commit=start)
                info.changelog = info.changelog.replace('%%', '')
                info.short_log = list(set(re.findall(r".*?([A-Z]+-\d+).*?", info.changelog)))

                if info.short_log:
                    search_parameters = {
                        "key": info.short_log,
                        "tags": [info.name, "common"]
                    }
                    info.linked_log = [issue.key for issue in st_client.issues.find(filter=search_parameters)]

    @staticmethod
    def create_release_issues(task, release):
        # Сортируем и группируем по параметрам тикета
        def key_fields(info):
            return info.release_issue_queue, info.release_issue_component, tuple(info.release_issue_followers or [])

        daemons_info = sorted(release.daemons_info, key=key_fields)
        issues_parameters = dict((key, list(group)) for key, group in groupby(daemons_info, key_fields))

        version_without_patch = ReleaseHelper.__get_version_without_patch(release.release_version)

        issues = []
        for issue_parameters, daemons_release_info in issues_parameters.items():
            search_parameters = {
                "queue": issue_parameters[0],
                "components": issue_parameters[1],
                "tags": "metrika-java-release-{}".format(version_without_patch),
                "type": "release"
            }
            followers = list(issue_parameters[2])
            if followers:
                search_parameters["followers"] = followers
            create_parameters = search_parameters.copy()
            create_parameters.update({
                "summary": "Релиз {} {}".format(ReleaseHelper._get_daemon_list(daemons_release_info), version_without_patch),
                "assignee": ReleaseHelper.__get_assignee(release),
                "description": ReleaseHelper.__get_daemons_description(daemons_release_info)
            })

            issue = ReleaseHelper.find_or_create_issue(task.st_client, search_parameters, create_parameters, task.id)
            logger.info("Issue {} created.".format(issue.key))
            for info in daemons_release_info:
                info.release_issue_key = issue.key

            table = "\n".join("|| {} | {} | {} | {} ||".format(
                "**{}**".format(info.name),
                "<{{**Связанные тикеты**\n{}\n}}>".format("\n".join(info.linked_log) if info.linked_log else "Отсутствуют"),
                "<{{**Все тикеты**\n{}\n}}>".format("\n".join("st:" + issue for issue in info.short_log) if info.short_log else "Отсутствуют"),
                "<{{**Лог**\n%%\n{}\n%%\n}}>".format(info.changelog or "Не содержит коммитов")
            ) for info in daemons_release_info)

            issue.comments.create(text="**!!(green)Версия {}!!**\n\n#|\n{}\n|#".format(release.release_version, table))
            issues.append(issue.key)
        return issues

    @staticmethod
    def get_build_tasks(task, author, all_programs, daemon_list, version):
        return metrika_java_helper.MetrikaJavaHelper.get_build_tasks(
            task,
            author,
            all_programs,
            daemon_list,
            version,
            ReleaseHelper.get_arcadia_url(ReleaseHelper.get_release_branch(version))
        )

    @staticmethod
    def create_tests_issues(task, settings, release):
        version_without_patch = ReleaseHelper.__get_version_without_patch(release.release_version)

        for info in (info for info in release.daemons_info if info.under_auto_tests):
            search_parameters = {
                "queue": "METRIQA",
                "components": 30311,
                "tags": "{}={}".format(info.name, version_without_patch)
            }
            create_parameters = search_parameters.copy()
            create_parameters.update({
                "summary": "Анализ логов тестов {} (Релиз {})".format(info.name, version_without_patch),
                "assignee": ReleaseHelper.__get_assignee(release),
                "description": "**Результаты выполнения автотестов для релиза {}**".format(info.release_issue_key)
            })
            issue = ReleaseHelper.find_or_create_issue(task.st_client, search_parameters, create_parameters, task.id)
            logger.info("Issue {} created.".format(issue.key))
            info.tests_issue_key = issue.key

    @staticmethod
    def __get_version_without_patch(release_version):
        major, minor, patch = version.StrictVersion(release_version).version
        return "{}.{}".format(major, minor)

    @staticmethod
    def __get_version_patch(release_version):
        major, minor, patch = version.StrictVersion(release_version).version
        return patch

    @staticmethod
    def __decrease_version_patch(release_version):
        major, minor, patch = version.StrictVersion(release_version).version
        patch -= 1
        return ReleaseHelper.build_version(major, minor, patch)

    @staticmethod
    def __get_daemons_description(daemons):
        return "#|\n{}|#\n".format("".join([ReleaseHelper.__get_daemon_description(daemon) for daemon in daemons]))

    @staticmethod
    def __get_daemon_description(daemon):
        daemon_template = "|| **{}** | <#<b>{}</b>#> | <#<b>{}</b>#> ||\n"
        testable = "<a style=\"color: red;\">Подлежит тестированию</a>" if daemon.under_tests else "<a style=\"color: green;\">Не подлежит тестированию</a>"
        auto_testable = "<a style=\"color: red;\">Автотесты в наличии</a>" if daemon.under_auto_tests else "<a style=\"color: green;\">Автотесты отсутствуют</a>"
        return daemon_template.format(daemon.name, testable, auto_testable)

    @staticmethod
    def __get_assignee(release):
        if "robot" in release.author:
            return utils.get_duty("metrika", "api-tests")
        else:
            return release.author

    @staticmethod
    def set_create_release_branch_info(task, branch):
        create_release_branch_info = "Ветка:<br/><a href=\"https://a.yandex-team.ru/arc_vcs/branches/?filterName={0}\">{0}</a>".format(branch)

        task.set_info(create_release_branch_info, do_escape=False)

    @staticmethod
    def set_create_release_issues_info(task, daemons_info):
        template = "<a href=\"https://st.yandex-team.ru/{0}\">{0}</a>: {1}"

        issues_info = dict((key, list(group)) for key, group in groupby(daemons_info, lambda info: (
            info.release_issue_key)))

        create_release_issue_info = "Задачи:<br/>" + "<br/>".join(
            [template.format(key, ", ".join([info.name for info in daemon_info_list])) for key, daemon_info_list in issues_info.items()])

        task.set_info(create_release_issue_info, do_escape=False)

    @staticmethod
    def set_create_tests_issues_info(task, daemons_info):
        daemons_info_with_tests_issue_key = [info for info in daemons_info if info.tests_issue_key]

        if daemons_info_with_tests_issue_key:
            template = "<a href=\"https://st.yandex-team.ru/{0}\">{0}</a>: {1}"

            create_tests_issues_info = "Задачи на анализ тестов:<br/>" + "<br/>".join(template.format(info.tests_issue_key, info.name) for info in daemons_info_with_tests_issue_key)

            task.set_info(create_tests_issues_info, do_escape=False)
