# coding=utf-8
import os
import logging
import re
from datetime import datetime

from sandbox import sdk2
from sandbox.common.types import task as task_type
from sandbox.common.types.misc import NotExists
from sandbox.common.utils import get_task_link
from sandbox.projects.common import constants
from sandbox.projects.common.build.YaPackage2 import YaPackage2Parameters
from sandbox.projects.common.build.ya_package_config.consts import PackageType
from sandbox.projects.common.decorators import retries
from sandbox.projects.metrika.admins import programs_list
from sandbox.projects.metrika.admins.cosmos import cosmos_launcher
from sandbox.projects.metrika.core import metrika_core_arcadia_tests_run
from sandbox.projects.metrika.utils import arcanum_api
from sandbox.projects.metrika.utils import resource_types
from sandbox.projects.metrika.utils import settings
from sandbox.projects.metrika.utils.pipeline import pipeline_errors
from sandbox.projects.metrika.utils.task_types import PRIVATE_YT_STORE_PARAMS
from sandbox.projects.metrika.utils.task_types.ya_package import MetrikaYaPackage
from sandbox.sdk2.vcs import svn


class MetrikaJavaHelper(object):

    @staticmethod
    def _green(text):
        return "<b style=\"color: #18a651;\">{}</b>".format(text)

    @staticmethod
    def _red(text):
        return "<b style=\"color: #ff4f49;\">{}</b>".format(text)

    @staticmethod
    def _blue(text):
        return "<b style=\"color: #1daff0;\">{}</b>".format(text)

    @staticmethod
    def _grey(text):
        return "<b style=\"color: #ababab;\">{}</b>".format(text)

    @classmethod
    def update_description(cls, task, author, daemon_list, arcadia_url):
        task.Parameters.description = cls._get_description(task.Parameters.description, author, daemon_list, arcadia_url)

    @classmethod
    def _get_description(cls, description, author, daemon_list, arcadia_url):
        return MetrikaJavaHelper._format_description(description, "{} из {} от {}".format(
            ", ".join(MetrikaJavaHelper._green(daemon) for daemon in daemon_list),
            MetrikaJavaHelper._grey(arcadia_url),
            MetrikaJavaHelper._blue(author)
        ))

    @staticmethod
    def _format_description(description, postfix, id="auto", separator="\n\n"):
        auto_div_start = "<div class=\"{}\">".format(id)
        auto_div_end = "</div>"
        auto_div = auto_div_start + postfix + auto_div_end

        if auto_div_start not in description:
            description += separator + auto_div
        else:
            description = re.sub(r"(?s){}.*{}".format(auto_div_start, auto_div_end), auto_div, description)

        return description

    @staticmethod
    def _update_description(task, postfix, id="auto", separator="\n\n"):
        task.Parameters.description = MetrikaJavaHelper._format_description(task.Parameters.description, postfix, id, separator)

    @staticmethod
    def _get_daemon_list(daemons_info):
        return " ".join([info.name for info in daemons_info])

    @staticmethod
    def get_arcadia_url(revision):
        return "{}:/#{}".format(svn.Arcadia.ARCADIA_ARC_SCHEME, revision)

    @staticmethod
    def get_all_programs(arcadia_url, path, replace_suffix=True):
        return programs_list.MetrikaProgramsList.get_all_programs(arcadia_url, path, replace_suffix)

    @staticmethod
    def get_build_tasks(task, author, all_programs, daemon_list, version, arcadia_url):
        MetrikaJavaHelper._check_buildability(all_programs, daemon_list)

        daemons_with_packages = [daemon for daemon in daemon_list if programs_list.PACKAGE_JSON in all_programs.get(daemon)]
        package_jsons = [all_programs.get(daemon).get(programs_list.PACKAGE_JSON) for daemon in daemons_with_packages]
        daemons_with_images = [daemon for daemon in daemon_list if programs_list.DOCKER_JSON in all_programs.get(daemon)]
        docker_jsons = [all_programs.get(daemon).get(programs_list.DOCKER_JSON) for daemon in daemons_with_images]

        build_tasks = []
        common_params = {
            YaPackage2Parameters.checkout_arcadia_from_url: arcadia_url,
            YaPackage2Parameters.build_type: constants.PROFILE_BUILD_TYPE,
            YaPackage2Parameters.custom_version: version,
            YaPackage2Parameters.run_tests: False,
            YaPackage2Parameters.run_long_tests: False,
            YaPackage2Parameters.clear_build: True,
            YaPackage2Parameters.semi_clear_build: True,
            YaPackage2Parameters.resource_type: resource_types.BaseMetrikaBinaryResource.name,
            YaPackage2Parameters.yp_token_vault: settings.yp_token,
            YaPackage2Parameters.release_to_ya_deploy: True
        }
        common_params.update(PRIVATE_YT_STORE_PARAMS)
        if package_jsons:
            params = {
                "description": MetrikaJavaHelper._get_description("Debian сборка java-демонов Метрики", author, daemons_with_packages, arcadia_url),
                YaPackage2Parameters.package_type: PackageType.DEBIAN.value,
                YaPackage2Parameters.packages: ";".join(package_jsons),
                YaPackage2Parameters.key_user: "robot-metrika-admin",
                YaPackage2Parameters.publish_package: True,
                YaPackage2Parameters.publish_to: "metrika-common",
                YaPackage2Parameters.dupload_max_attempts: 7,
            }
            build_tasks.append((MetrikaYaPackage, dict(common_params, **params)))
        if docker_jsons:
            params = {
                "description": MetrikaJavaHelper._get_description("Docker сборка java-демонов Метрики", author, daemons_with_images, arcadia_url),
                YaPackage2Parameters.package_type: PackageType.DOCKER.value,
                YaPackage2Parameters.packages: ";".join(docker_jsons),
                YaPackage2Parameters.docker_image_repository: "metrika/java",
                YaPackage2Parameters.docker_push_image: True,
                YaPackage2Parameters.docker_user: settings.login,
                YaPackage2Parameters.docker_token_vault_name: settings.docker_registry_token,
            }
            build_tasks.append((MetrikaYaPackage, dict(common_params, **params)))
        return build_tasks

    @staticmethod
    def _check_buildability(all_programs, daemon_list):
        for daemon in daemon_list:
            if not all_programs.get(daemon):
                raise pipeline_errors.PipelineError(
                    "Для демона {} не найдены {} и {} (возможна опечатка в имени или незакоммиченные изменения)".format(
                        daemon,
                        programs_list.PACKAGE_JSON,
                        programs_list.DOCKER_JSON,
                    )
                )

    @staticmethod
    def wait_tasks(task_ids):
        if task_ids:
            raise sdk2.WaitTask(task_ids, task_type.Status.Group.FINISH + task_type.Status.Group.BREAK)

    @staticmethod
    def check_task(parent, task_id):
        task = parent.find(id=task_id).first()

        if task.status not in task_type.Status.Group.SUCCEED:
            raise pipeline_errors.PipelineError("Дочерняя таска {} завершилась с ошибкой, перезапустите...".format(task.type))

    @staticmethod
    def get_tests_tasks(daemons_info, arcadia_url, all_programs, report_ttl, tracker_issue=None, is_release=False):
        tests_tasks = []
        for info in daemons_info:
            tests_issue_key = tracker_issue
            if hasattr(info, "tests_issue_key"):
                tests_issue_key = tests_issue_key or info.tests_issue_key

            if info.under_auto_tests:
                tests_tasks.append((
                    cosmos_launcher.CosmosLauncher,
                    dict(
                        description="Запуск тестов {} из {}".format(info.name, arcadia_url),
                        vcs="arcadia",
                        arcadia_url=arcadia_url,
                        arcadia_path="metrika/java/tests",
                        build_configuration=info.tests_build_configuration,
                        packs_dirs=["{}/packs".format(info.name)],
                        run_configuration=info.tests_run_configuration,
                        report_startrek=bool(tests_issue_key),
                        issue_key=tests_issue_key,
                        semaphore_name="COSMOS_LAUNCHER-{name}".format(name=info.name) if info.need_semaphore else None,
                        report_ttl=report_ttl
                    )
                ))

            if is_release:
                arcanum = arcanum_api.ArcanumApi(token=sdk2.Vault.data(settings.owner, settings.arcanum_token))
                daemon_path = os.path.dirname(all_programs.get(info.name).values()[0])
                if "ut_medium" in arcanum.get_nodes(daemon_path):
                    tests_tasks.append((
                        metrika_core_arcadia_tests_run.MetrikaCoreArcadiaTestsRun,
                        dict(
                            description="Запуск аркадийных тестов {} из {}".format(info.name, arcadia_url),
                            checkout_arcadia_from_url=arcadia_url,
                            targets=os.path.join(daemon_path, "ut_medium"),
                            fail_task_on_test_failure=True,
                            report_startrek=bool(tests_issue_key),
                            issue_key=tests_issue_key,
                            allure_report_ttl=report_ttl,
                        )
                    ))

        return tests_tasks

    @staticmethod
    def get_stable_version(daemon_name, raise_for_fail=True):
        return MetrikaJavaHelper.get_version(daemon_name, "production", raise_for_fail)

    @staticmethod
    @retries(3)
    def get_version(daemon_name, env, raise_for_fail=True):
        version = MetrikaJavaHelper.get_package_version(MetrikaJavaHelper.get_package_name(daemon_name), env)
        if version is None and raise_for_fail:
            raise pipeline_errors.PipelineError("Для демона '{}' не найдено версий в окружении '{}'.".format(daemon_name, env))

        return version

    @staticmethod
    def get_stable_package_version(package_name):
        return MetrikaJavaHelper.get_package_version(package_name, "production")

    @staticmethod
    @retries(3)
    def get_package_version(package_name, env):
        from metrika.pylib.mtapi.packages import PackagesAPI, BadRequestException

        mtapi_packages = PackagesAPI()
        mtapi_packages._verify = False

        try:
            return mtapi_packages.pkg_version_per_environment(env=env, pkg_name=package_name)[0].pkg_version
        except BadRequestException:
            logging.exception("Version for '{}' was not found.".format(package_name))
            return
        except Exception as err:
            logging.exception("Unhandled exception during mtapi_packages.pkg_version_per_environment(env='{}', pkg_name='{}') call: {}".format(env, package_name, err))
            raise pipeline_errors.PipelineError("Не удалось выполнить запрос в mtapi для поиска стабильной версии '{}'.".format(package_name))

    @staticmethod
    @retries(3)
    def get_package_version_on_cluster(package_name, cluster_name):
        from metrika.pylib.mtapi.packages import PackagesAPI, BadRequestException

        mtapi_packages = PackagesAPI()
        mtapi_packages._verify = False

        try:
            return mtapi_packages.pkg_version_per_group(
                root_group=cluster_name,
                pkg_name=package_name
            )
        except BadRequestException:
            raise pipeline_errors.PipelineError("Для пакета '{}' не найдено версий на '{}'.".format(package_name, cluster_name))

    @staticmethod
    def get_package_name(daemon_name):
        return "{}-metrika-yandex".format(daemon_name)

    @staticmethod
    def find_or_create_issue(st_client, search_parameters, create_parameters, task_id=None, sprint_board=None):
        issues = st_client.issues.find(filter=search_parameters)
        issue = None

        if issues:
            try:
                issue = next(iter(issues))
            except StopIteration:
                logging.info("No issues were found.")

        if issue:
            if issue.status.key not in ["open", "in_progress", "testing"]:
                reopen_transition = next(iter([t for t in issue.transitions.get_all() if t.to.key == 'open']), None)
                if reopen_transition:
                    reopen_transition.execute()
        else:
            if sprint_board:
                for sprint in st_client.boards[sprint_board].sprints:
                    if datetime.strptime(sprint.startDate, '%Y-%m-%d') <= datetime.now() <= datetime.strptime(sprint.endDate, '%Y-%m-%d'):
                        create_parameters['sprint'] = sprint.id
                        break

            issue = st_client.issues.create(**create_parameters)

        if task_id:
            from startrek_client import exceptions
            try:
                issue.remotelinks.create("relates", task_id, "ru.yandex.sandbox")
            except exceptions.StartrekServerError as e:
                if 'уже связаны' not in str(e):
                    raise

        return issue

    @staticmethod
    def set_build_packages_info(task, daemon_dictionary, repo="metrika-common"):
        template = "<a href=\"https://dist.yandex.ru/find?pkg={0}&repo={2}&ver={1}&human=True\">{0}={1}</a>"

        build_packages_info = "Пакеты:<br/>" + "<br/>".join(
            [template.format(MetrikaJavaHelper.get_package_name(daemon), version, repo) for daemon, version in
             daemon_dictionary.iteritems()]
        )

        task.set_info(build_packages_info, do_escape=False)

    @staticmethod
    def set_build_info(task, task_ids, repo="metrika-common"):
        packages = []
        images = []
        tarballs = []
        for task_id in task_ids:
            build_task = sdk2.Task[task_id]
            resources = [{"name": resource.resource_name, "version": resource.resource_version, "id": resource.id}
                         for resource in sdk2.Resource.find(task_id=task_id, type=resource_types.BaseMetrikaBinaryResource).limit(256)]
            if build_task.Parameters.package_type == PackageType.DEBIAN.value:
                packages.extend(resources)
            if build_task.Parameters.package_type == PackageType.DOCKER.value:
                images.extend(resources)
            if build_task.Parameters.package_type == PackageType.TARBALL.value:
                tarballs.extend(resources)

        package_info = None
        if packages:
            package_template = "<a href=\"https://dist.yandex.ru/find?pkg={name}&repo={repo}&ver={version}&human=True\">{name}={version}</a>"
            package_info = "Пакеты:<br/>" + "<br/>".join(
                [package_template.format(repo=repo, **package) for package in packages]
            )

        image_info = None
        if images:
            image_template = "<a href=\"https://{version}\">{name}={version}</a>"
            image_info = "Образы:<br/>" + "<br/>".join(
                [image_template.format(**image) for image in images]
            )

        tarball_info = None
        if tarballs:
            tarball_template = "<a href=\"https://sandbox.yandex-team.ru/resource/{id}\">{name}={version}</a>"
            tarball_info = "Тарболы:<br/>" + "<br/>".join(
                [tarball_template.format(**tarball) for tarball in tarballs]
            )

        task.set_info("<br/>".join(filter(None, [package_info, image_info, tarball_info])), do_escape=False)

    @staticmethod
    def set_create_deploy_test_stands_info(task, daemons_deploy_test_stands):
        if daemons_deploy_test_stands:
            create_deploy_test_stands_info = "Автобеты:<br/>" + "<br/>".join(
                set([MetrikaJavaHelper.get_deploy_test_stands_info(deploy_test_stand.daemon_name, deploy_test_stand.name, deploy_test_stand.bishop_environment_prefix)
                     for deploy_test_stand in daemons_deploy_test_stands])
            )

            task.set_info(create_deploy_test_stands_info, do_escape=False)

    @staticmethod
    def get_deploy_test_stands_info(daemon_name, name, bishop_environment_prefix):
        deploy_link = "<a href=\"https://deploy.yandex-team.ru/stage/{0}-autobeta-{1}\">{0}-autobeta-{1}</a>".format(daemon_name, name)

        bishop_parent = "{}.{}".format(bishop_environment_prefix, daemon_name) if bishop_environment_prefix else daemon_name
        bishop_link = "(<a href=\"https://bishop.mtrs.yandex-team.ru/environment/metrika.deploy.java.testing.autobetas.{0}.{1}\">Bishop</a>)".format(bishop_parent, name)

        return "{} {}".format(deploy_link, bishop_link)

    @staticmethod
    def set_run_tests_info(task, test_tasks_ids):
        if test_tasks_ids:
            template = "<a href=\"{}\">{} #{}</a>"

            run_tests_info = "Тесты:<br/>" + "<br/>".join(template.format(
                get_task_link(task_id),
                sdk2.Task[task_id].type.name,
                task_id
            ) for task_id in test_tasks_ids)

            task.set_info(run_tests_info, do_escape=False)

    @classmethod
    def check_daemons(cls, task, daemons_list):
        if task.Context._can_check_daemons_list == NotExists:
            task.Context._can_check_daemons_list = False
            return
        elif not task.Context._can_check_daemons_list:
            task.Context._can_check_daemons_list = True
            return

        if not daemons_list:
            raise ValueError('Список демонов не может быть пустым')

        for daemon in daemons_list:
            if ',' in daemon:
                raise ValueError('Демона должны быть указаны в списке, а не через запятую (нажмите "+" у параметра)')
