import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_resource_link
from sandbox.projects.sandbox.remote_copy_resource_2 import RemoteCopyResource2


class FlutterResourceTask(sdk2.Task):
    """
        Creates resource with Flutter of given version
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("Platform") as platform:
            platform.values["macos"] = platform.Value("MacOS")
            platform.values["linux"] = platform.Value("Linux")
        version = sdk2.parameters.String('Flutter version',
                                         description='Available versions: '
                                                     'https://docs.flutter.dev/development/tools/sdk/releases',
                                         required=True)
        type = sdk2.parameters.String('Version type', default="stable", required=True)
        url = sdk2.parameters.String('Remote resource',
                                     description="Url for remote resource with flutter. "
                                                 "If not set, storage.googleapis.com url is automatically generated")

        with sdk2.parameters.Output():
            result_resource_id = sdk2.parameters.Resource("Flutter resource")

    class Context(sdk2.Task.Context):
        task_id = None
        flutter_resource_id = None

    def on_execute(self):
        with self.memoize_stage.create_reource:
            task = self._create_task()
            self.Context.task_id = task.id

            task.enqueue()
            raise sdk2.WaitTask(task, ctt.Status.Group.FINISH)

        task = sdk2.Task[self.Context.task_id]
        if not task or task.status not in ctt.Status.Group.SUCCEED:
            raise TaskFailure("Failed create resource. See subtask")

        self.Parameters.result_resource_id = task.Parameters.result_resource_id

    def _create_task(self):
        params = {
            RemoteCopyResource2.Parameters.resource_arch.name: self._get_resource_arch(),
            RemoteCopyResource2.Parameters.resource_type.name: "FLUTTER",
            RemoteCopyResource2.Parameters.created_resource_name.name: self._get_created_resource_name(),
            RemoteCopyResource2.Parameters.remote_file_name.name: self._get_remote_file_name(),
            RemoteCopyResource2.Parameters.remote_file_protocol.name: "http",
            RemoteCopyResource2.Parameters.store_forever.name: True,
            RemoteCopyResource2.Parameters.resource_attrs.name: self._get_resource_attrs(),
        }
        return RemoteCopyResource2(
            self,
            description="Create Flutter resource with version {} and type {}".format(
                self.Parameters.version, self.Parameters.type),
            priority=min(
                self.Parameters.priority,
                ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)  # default API limitation
            ),
            **params
        )

    def _get_created_resource_name(self):
        if self.Parameters.url:
            return self.Parameters.url.split("/")[-1]
        if self.Parameters.platform == "macos":
            return "flutter.zip"
        if self.Parameters.platform == "linux":
            return "flutter.tar.xz"

    def _get_resource_arch(self):
        if self.Parameters.platform == "macos":
            return "osx"
        if self.Parameters.platform == "linux":
            return "linux"
        return "any"

    # https://storage.googleapis.com/flutter_infra_release/releases/stable/macos/flutter_macos_2.8.1-stable.zip
    # https://storage.googleapis.com/flutter_infra_release/releases/stable/linux/flutter_linux_2.8.1-stable.tar.xz
    def _get_remote_file_name(self):
        if self.Parameters.url:
            return self.Parameters.url
        file_ext = "zip" if self.Parameters.platform == "macos" else "tar.xz"
        return "https://storage.googleapis.com/flutter_infra_release/releases/{0}/{1}/flutter_{1}_{2}-{0}.{3}".format(
            self.Parameters.type, self.Parameters.platform, self.Parameters.version, file_ext
        )

    def _get_resource_attrs(self):
        return "version={}, version_type={}".format(
            self.Parameters.version, self.Parameters.type
        )

    @sdk2.header()
    def header(self):
        header = ''
        if self.Context.flutter_resource_id:
            header += 'Flutter: <a href="{}">{}</a>\n'.format(
                get_resource_link(self.Context.flutter_resource_id), self.Context.flutter_resource_id
            )
        return '<span style="font-size: 15px;">{}</span>'.format(header)
