import logging

import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag
from sandbox.common.utils import get_resource_link
from sandbox.projects.metrika.mobile.ios.simulator.simulator_environment import IosSimulator, IosSimulatorEnvironment
from sandbox.projects.mobile_apps.utils.rvm_plus_ruby_env import RvmPlusRubyEnvironment
from sandbox.sdk2.environments import Xcode
from sandbox.projects.metrika.utils import custom_report_logger

_logger = logging.getLogger('ios_simulator_environment_preparer')


class IosSimulatorEnvironmentPreparer(sdk2.Task):
    """
        Creates resource with iOS Simulator environment of given version
    """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        environments = [RvmPlusRubyEnvironment()]

    class Parameters(sdk2.Task.Parameters):
        use_catalina = sdk2.parameters.Bool("Use Mac OS Catalina", default=True)
        xcode_version = sdk2.parameters.String("Xcode version",
                                               description="Available versions: https://nda.ya.ru/t/9_Kdb0_y3mKvk7",
                                               required=True)
        simulator_version = sdk2.parameters.String("Simulator version", required=True)

    class Context(sdk2.Task.Context):
        resource_id = None

    def on_save(self):
        self.Requirements.client_tags = (Tag.MOBILE_MONOREPO | Tag.CUSTOM_MOBILE_MONOREPO) & Tag.USER_MONOREPO
        if self.Parameters.use_catalina:
            self.Requirements.client_tags = self.Requirements.client_tags & Tag.OSX_CATALINA

    def on_prepare(self):
        with sdk2.helpers.ProgressMeter('Xcode {} prepare'.format(self.Parameters.xcode_version)):
            Xcode(self.Parameters.xcode_version).prepare()

    def on_execute(self):
        with sdk2.helpers.ProgressMeter('Install xcode-install'):
            self._cmd('gem install xcode-install', 'install_xcode-install',
                      'Failed install xcode-install. See common.log.')
        with sdk2.helpers.ProgressMeter('Install simulator iOS {}'.format(self.Parameters.simulator_version)):
            self._cmd('xcversion simulators --install="iOS {}"'.format(self.Parameters.simulator_version),
                      'install_simulator',
                      'Failed install simulator iOS {}. See common.log.'.format(self.Parameters.simulator_version))
            self._cmd('xcrun simctl list', 'print_simulators', 'Failed print simulators')
        with sdk2.helpers.ProgressMeter('Pack simulator'):
            self._pack_simulator(self._work_dir('simulator.tar.gz'))
        with sdk2.helpers.ProgressMeter('Create Resource'):
            self._create_resource(self._work_dir('simulator.tar.gz'))

    def _work_dir(self, *path):
        return str(self.path(*path))

    def _pack(self, dest):
        sim_path = IosSimulatorEnvironment.simulator_path(self.Parameters.simulator_version)
        _logger.info('Pack iOS Simulator from {} to {}'.format(sim_path, dest))
        self._cmd(
            'tar czf {} -C "{}" .'.format(dest, sim_path),
            'pack_simulator',
            'Can not pack iOS Simulator. See common.log.'
        )

    def _create_resource(self, tar_path):
        ios_simulator = IosSimulator(
            self, 'iOS v{}'.format(self.Parameters.simulator_version), tar_path, ctm.OSFamily.OSX,
            version=self.Parameters.simulator_version,
            platform='darwin'
        )
        self.Context.resource_id = ios_simulator.id
        sdk2.ResourceData(ios_simulator).ready()

    def _cmd(self, args, logger_name, message):
        _logger.info('CMD: {}'.format(args))
        cmd_process_log = sdk2.helpers.process.ProcessLog(logger=logging.getLogger(logger_name))
        rc = sdk2.helpers.process.subprocess.Popen(
            args, shell=True, env=self.env, stdout=cmd_process_log.stdout, stderr=cmd_process_log.stderr
        ).wait()
        cmd_process_log.close()
        _logger.info('The command executed.')
        if rc:
            raise TaskFailure('{} {}'.format(message, rc))

    @sdk2.header()
    @custom_report_logger
    def header(self):
        header = ''
        if self.Context.resource_id:
            header += 'Resource: <a href="{}">{}</a>\n'.format(
                get_resource_link(self.Context.resource_id), self.Context.resource_id
            )
        return '<span style="font-size: 15px;">{}</span>'.format(header)

    @property
    def env(self):
        try:
            return self._env
        except AttributeError:
            self._env = dict(_.split('=', 1) for _ in sdk2.helpers.subprocess.check_output(
                ['/bin/bash', '-c', '. /etc/profile && printenv']).splitlines())
        return self._env
