import logging
import os

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.utils import get_resource_link
from sandbox.sdk2.environments import Xcode

from sandbox.projects.metrika.mobile.ios.symbols.system_ios_symbols import SystemIosSymbols
from sandbox.projects.metrika.mobile.sdk.helpers.ShellExecutor import ShellExecutor
from sandbox.projects.metrika.utils import custom_report_logger

MOBMET_INTAPI_TVM_ID_PROD = 2002840

IPSW_FILE = 'iphone.ipsw'
IPSW_EXTRACT_DIR = 'ipsw'
SYMBOLS_DIR = 'symbols'

_logger = logging.getLogger('upload_system_ios_symbols')


class UploadSystemIosSymbols(sdk2.Task):
    """
        see https://github.com/Zuikyo/iOS-System-Symbols#extract-symbols-from-firmware
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = (ctc.Tag.MOBILE_MONOREPO | ctc.Tag.CUSTOM_MOBILE_MONOREPO) & ctc.Tag.USER_MONOREPO
        dns = ctm.DnsType.DNS64
        disk_space = 60 * 1024  # Xcode 13 requires 45 Gb (https://developer.apple.com/forums/thread/693336)

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group('Binary task parameters') as binary_task_params:
            with sdk2.parameters.RadioGroup('Type') as release_type:
                release_type.values.stable = release_type.Value(value='stable', default=True)
                release_type.values.test = release_type.Value(value='test')
                release_type.values.custom = release_type.Value(value='custom')
            release_version = sdk2.parameters.String('Version', required=False)
        with sdk2.parameters.Group('Parameters') as params:
            ipsw_url = sdk2.parameters.String('ipsw url', required=True)
            xcode_version = sdk2.parameters.String('Xcode version')
            dsc_extractor_resource = sdk2.parameters.Resource('Resource with dsc_extractor.bundle')
        with sdk2.parameters.Group("Action") as action:
            create_resource = sdk2.parameters.Bool('Create sandbox resource',
                                                   description='Create resource with zip archive',
                                                   default=True)
            upload_to_appmetrica = sdk2.parameters.Bool('Upload to AppMetrica', default=False)

    class Utils:
        shell = ShellExecutor()

    class Context(sdk2.Task.Context):
        dsym_resource_id = None
        appmetrica_crash_dsym_id = None

    def on_save(self):
        if self.Parameters.release_type == 'custom':
            return
        else:
            attrs = {'target': 'upload_system_ios_symbols', 'release': self.Parameters.release_type}
            if self.Parameters.release_version:
                attrs['version'] = self.Parameters.release_version
            self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(attrs=attrs).first().id

    def on_prepare(self):
        if self.Parameters.xcode_version:
            with sdk2.helpers.ProgressMeter("Xcode {} prepare".format(self.Parameters.xcode_version)):
                Xcode(self.Parameters.xcode_version).prepare()

    def on_execute(self):
        from metrika.sdk.upload_ios_symbols.appmetrica_dsym_uploader import AppMetricaDsymUploader
        from metrika.sdk.upload_ios_symbols.developer_apple_download_helper import DeveloperAppleDownloadHelper
        from metrika.sdk.upload_ios_symbols.dmg import Dmg, DmgMountException
        from metrika.sdk.upload_ios_symbols.ipsw_plist import IpswPlist
        import metrika.sdk.upload_ios_symbols.utils as utils

        with sdk2.helpers.ProgressMeter("Download ipsw") as pm:
            dad_helper = DeveloperAppleDownloadHelper()
            # TODO: (METRIKALIB-8202) login in developer.apple.com
            _logger.info("Downloading ipsw from {}".format(self.Parameters.ipsw_url))
            _logger.info("Saving to {}".format(IPSW_FILE))
            dad_helper.download(self.Parameters.ipsw_url, IPSW_FILE, pm)
        with sdk2.helpers.ProgressMeter('Extract ipsw') as pm:
            _logger.info("Extract ipsw {}".format(IPSW_FILE))
            _logger.info("Save to {}".format(IPSW_EXTRACT_DIR))
            utils.extract_ipsw(IPSW_FILE, IPSW_EXTRACT_DIR, pm)

        _logger.info("Dmg search in {}".format(IPSW_EXTRACT_DIR))
        dmgs = utils.find_dmgs(IPSW_EXTRACT_DIR)
        _logger.info("Found dmgs: {}".format(dmgs))

        arch_list = set()
        symbols_dirs = []
        with sdk2.helpers.ProgressMeter("Processing dmgs", maxval=len(dmgs)) as dmgs_pm:
            dsc_extractor = self._get_dsc_extractor()
            for dmg in dmgs:
                dmg_name = os.path.basename(dmg)
                dmgs_pm.message = "Processing dmgs: {}".format(dmg_name)
                _logger.info("Processing dmgs: {}".format(dmg_name))
                try:
                    with Dmg(dmg) as dmg_mount:
                        _logger.info("Dylds search in {}".format(dmg_mount))
                        dylds = utils.find_dylds(dmg_mount)
                        _logger.info("Found dylds: {}".format(dylds))
                        with sdk2.helpers.ProgressMeter("Extract dylds", maxval=len(dylds)) as dylds_pm:
                            for dyld in dylds:
                                arch_list.add(os.path.basename(dyld)[len("dyld_shared_cache_"):].split(".")[0])
                                extract_dir = os.path.join(SYMBOLS_DIR, dmg_name, os.path.basename(dyld))
                                symbols_dirs.append(extract_dir)
                                dmgs_pm.message = "Extract dylds: {}".format(dyld)
                                _logger.info("Extract dylds: {}".format(dyld))
                                _logger.info("Saving to {}".format(extract_dir))
                                dsc_extractor.extract(dyld, extract_dir)
                                dylds_pm.add(1)
                except DmgMountException:
                    pass  # skip broke dmg
                dmgs_pm.add(1)
            _logger.info("Found arch {}".format(arch_list))

        plist = IpswPlist(IPSW_EXTRACT_DIR)
        merged_symbols_dir = self._get_symbol_dir_name(plist, arch_list)
        _logger.info("Merging symbols from {}".format(symbols_dirs))
        _logger.info("Saving to {}".format(merged_symbols_dir))
        with sdk2.helpers.ProgressMeter("Merge symbols") as pm:
            utils.merge_symbols(symbols_dirs, merged_symbols_dir, pm)

        archive_path = "{}.zip".format(merged_symbols_dir)
        _logger.info("Archiving symbols from {}".format(merged_symbols_dir))
        _logger.info("Saving to {}".format(archive_path))
        with sdk2.helpers.ProgressMeter("Pack symbols") as pm:
            utils.pack_symbols(merged_symbols_dir, archive_path, pm)
        if os.path.getsize(archive_path) < 100 * 1024 * 1024:
            raise Exception("Zip file with symbols is small in size: {}b".format(os.path.getsize(archive_path)))

        if self.Parameters.create_resource:
            with sdk2.helpers.ProgressMeter("Create resource"):
                self._create_resource(archive_path, plist, arch_list)
        if self.Parameters.upload_to_appmetrica:
            with sdk2.helpers.ProgressMeter("Upload to AppMetrica"):
                secret = sdk2.yav.Secret(AppMetricaDsymUploader.get_tvm_secret_version())
                client_secret = secret.data(auto_decode=True)["client_secret"]
                _logger.info("secret type: " + str(type(client_secret)))
                AppMetricaDsymUploader.upload(archive_path, client_secret=client_secret)

    def _get_dsc_extractor(self):
        from metrika.sdk.upload_ios_symbols.dsc_extractor import DscExtractor

        if self.Parameters.dsc_extractor_resource:
            resource = sdk2.Resource[self.Parameters.dsc_extractor_resource]
            data = sdk2.ResourceData(resource)
            return DscExtractor(str(data.path))
        else:
            return DscExtractor()

    def _create_resource(self, archive_path, plist, arch_list):
        resource = SystemIosSymbols(
            self, self._get_symbol_dir_name(plist, arch_list), archive_path, ctm.OSFamily.OSX,
            ios_version=plist.get_ios_version(),
            model_number=plist.get_model_number(),
            ios_arch=';'.join(arch_list),
            devices=plist.get_devices(),
        )

        self.Context.dsym_resource_id = resource.id
        sdk2.ResourceData(resource).ready()

    def _get_symbol_dir_name(self, plist, arch_list):
        return '{} ({}) {}'.format(plist.get_ios_version(), plist.get_model_number(), ' '.join(arch_list))

    @sdk2.header()
    @custom_report_logger
    def header(self):
        def tr(key, value):
            return '<tr><td>{}</td><td>{}</td></tr>'.format(key, value)

        def a(name, link):
            return '<a href="{}" target="_blank">{}</a>'.format(link, name)

        table = '<table class="cf__t"><tbody>'
        if self.Context.dsym_resource_id:
            table += tr('Resource', a(self.Context.dsym_resource_id, get_resource_link(self.Context.dsym_resource_id)))
        if self.Context.appmetrica_crash_dsym_id:
            table += tr('AppMetrica', self.Context.appmetrica_crash_dsym_id)
        table += '</tbody></table>'
        return table
