# coding=utf-8
import base64
import logging
import os

import jinja2

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.errors import InvalidResource
from sandbox.common.errors import TaskFailure
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.projects.metrika.mobile.sdk.helpers.GradleContainerFinder import GradleContainerFinder
from sandbox.projects.metrika.mobile.sdk.helpers.GradleExecutor import GradleExecutor
from sandbox.projects.metrika.mobile.sdk.helpers.ShellExecutor import ShellExecutor
from sandbox.projects.metrika.mobile.sdk.helpers.TarResourceHelper import TarResourceHelper
from sandbox.projects.metrika.mobile.sdk.helpers.TeamCityArtifactPublisher import TeamCityArtifactPublisher
from sandbox.projects.metrika.mobile.sdk.helpers.VcsHelper import VcsHelper
from sandbox.projects.metrika.mobile.sdk.parameters.GradleParameters import GradleParameters
from sandbox.projects.metrika.mobile.sdk.parameters.ResourceParameters import ResourceParameters
from sandbox.projects.metrika.mobile.sdk.parameters.TeamcityParameters import TeamcityParameters
from sandbox.projects.metrika.mobile.sdk.parameters.VcsParameters import VcsParameters
from sandbox.projects.common.decorators import retries
from sandbox.projects.metrika.utils import custom_report_logger

KEYRING_FILENAME = 'secring.gpg'


class AndroidGradleRunner(sdk2.Task):
    """
    """

    class Utils:
        vcs_helper = VcsHelper()
        shell_executor = ShellExecutor()
        teamcity_publisher = TeamCityArtifactPublisher()
        gradle_executor = GradleExecutor()
        tar_resource_helper = TarResourceHelper(shell_executor)
        gradle_container_finder = GradleContainerFinder()

    class Requirements(sdk2.Task.Requirements):
        client_tags = Tag.GENERIC
        dns = DnsType.DNS64
        disk_space = 32768
        ramdrive = ctm.RamDrive(ctm.RamDriveType.TMPFS, 2048, None)
        cores = 8
        ram = 4 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    class Parameters(sdk2.Task.Parameters):
        run_target = sdk2.parameters.List("Gradle target")
        vcs = VcsParameters
        gradle = GradleParameters

        with sdk2.parameters.Group("Signing") as android_signing_parameters:
            keyring_file_name = sdk2.parameters.String("Имя параметра для передачи keyringFile",
                                                       description="Передается как gradle-параметр")
            keyring_file_value = sdk2.parameters.Vault("Keyring file в base64",
                                                       description="Достается из valut и пишется на диск, "
                                                                   "задается в формате owner:name")

        teamcity = TeamcityParameters
        resource = ResourceParameters

        with sdk2.parameters.Group("Other") as other_parameters:
            teamcity_build_id = sdk2.parameters.String("Teamcity build ID",
                                                       description="ID сборки на teamcity")
            use_tmpfs = sdk2.parameters.Bool("Use tmpfs",
                                             description="Монтировать и использовать RamDrive",
                                             required=True, default=True)
            container = sdk2.parameters.Container("Environment container resource", default_value=1417403637)
            container_version = sdk2.parameters.String("[Not used] Container version", default="stable")
        with sdk2.parameters.Group("Experiments") as experiments_parameters:
            add_write_permissions = sdk2.parameters.Bool("Add write permission",
                                                         description="DON'T USE IT. NEVER. "
                                                                     "https://st.yandex-team.ru/FRANKENSTEIN-997",
                                                         default=False)

    class Context(sdk2.Task.Context):
        artifacts_resource_id = None

    def on_save(self):
        if self.parent is not None and hasattr(self.parent.Parameters, 'add_write_permissions'):
            self.Parameters.add_write_permissions = self.parent.Parameters.add_write_permissions

    def on_execute(self):
        with self.repo:
            try:
                gradle_parameters = self._get_gradle_properties()
                system_gradle_parameters = self._get_system_gradle_properties()
                statuses = []
                for tasks in self._get_gradle_tasks():
                    with sdk2.helpers.ProgressMeter("Running {}".format(tasks)):
                        exit_code = self.Utils.gradle_executor.execute_gradle(
                            path=self._work_dir(self.Parameters.gradle.gradle_base_dir),
                            tasks=[self.Parameters.gradle.gradle_log_level, "--stacktrace"] + tasks,
                            gradle_properties=gradle_parameters,
                            system_properties=system_gradle_parameters,
                            timeout=self.execute_shell_timeout,
                            build_file=self.Parameters.build_file,
                            environment=self.env)
                        statuses.append([tasks, exit_code])
                success = True
                messages = []
                for status in statuses:
                    if status[1]:
                        success = False
                        self.logger.error("Command\n{!r}\nFAILED. Exit code: {!r}".format(*status))
                        messages.append("Command {0} FAILED".format(status[0]))
                self._write_error_to_teamcity('\n'.join(messages))
                if not success:
                    raise TaskFailure("Some commands FAILED. See report above.")
                with sdk2.helpers.ProgressMeter("Save built resource"):
                    self._save_to_resource()
            except Exception:
                self.logger.error("Exception in scenario", exc_info=True)
                raise
            finally:
                with sdk2.helpers.ProgressMeter("Publish artifacts to teamcity"):
                    with self.memoize_stage.create_teamcity_artifacts(commit_on_entrance=False):
                        self.create_teamcity_artifacts()

    def _input_dir(self, *path):
        return str(self.path("input", *path))

    def _output_dir(self, *path):
        p = self.path("output", *path)
        if not p.exists():
            os.makedirs(str(p))
        return str(p)

    def _prepare_keyring_file(self):
        if self.Parameters.keyring_file_value:
            keyring = sdk2.Vault.data(self.Parameters.keyring_file_value.owner,
                                      self.Parameters.keyring_file_value.name)
            with open(self._work_dir(KEYRING_FILENAME), 'w') as keyring_file:
                keyring_file.write(base64.b64decode(keyring))

    def _prepare_input_resource(self):
        if self.Parameters.input_resource:
            if self.Parameters.input_resource_path:
                input_resource_path = self._work_dir(self.Parameters.input_resource_path)
            else:
                input_resource_path = self._input_dir()
            self.Utils.tar_resource_helper.extract_input_resource(self.Parameters.input_resource, input_resource_path, self)

    def _save_to_resource(self):
        if self.Parameters.output_resource_ttl:
            if self.Parameters.output_resource_path:
                output_resource_path = self._work_dir(self.Parameters.output_resource_path)
            else:
                output_resource_path = self._output_dir()
            try:
                self.Parameters.output_resource = self.Utils.tar_resource_helper.save_to_resource(
                    task=self,
                    description="Build resource",
                    current_dir=self.path(output_resource_path).parent,
                    resource_dir_name=self.path(output_resource_path).name,
                    ttl=self.Parameters.output_resource_ttl,
                    task_dir=self.path()
                )
                self.Context.artifacts_resource_id = self.Parameters.output_resource.id
            except InvalidResource:
                logging.info("Something wrong with resource. It may be empty")

    def _get_gradle_tasks(self):
        if self.Parameters.run_target:
            return [i.split() for i in self.Parameters.run_target]
        return []

    def _get_gradle_properties(self):
        gradle_properties = {
            "sandbox.teamcity.messages": self._get_teamcity_messages_path(),
        }
        if self.Parameters.teamcity_build_id:
            gradle_properties.update({
                "sandbox.build.number": self.Parameters.teamcity_build_id,
            })
        if self.Parameters.branch:
            gradle_properties.update({
                "sandbox.build.branch": self.Parameters.branch,
            })

        if not self.Parameters.input_resource_path and self.Parameters.input_resource_path_param_name:
            gradle_properties.update({
                self.Parameters.input_resource_path_param_name: self._input_dir()
            })

        if not self.Parameters.output_resource_path and self.Parameters.output_resource_path_param_name:
            gradle_properties.update({
                self.Parameters.output_resource_path_param_name: self._output_dir()
            })

        if self.Parameters.keyring_file_value:
            gradle_properties[self.Parameters.keyring_file_name] = self._work_dir(KEYRING_FILENAME)

        gradle_properties.update({k: v for k, v in self.Parameters.gradle.gradle_parameters.iteritems() if v})
        gradle_properties.update({k: sdk2.Vault.data(*v.split(':', 1))
                                  for k, v in self.Parameters.gradle.secret_gradle_parameters.iteritems() if v})
        return gradle_properties

    def _get_system_gradle_properties(self):
        gradle_properties = {}
        gradle_properties.update({k: v for k, v in self.Parameters.gradle.system_gradle_parameters.iteritems() if v})
        return gradle_properties

    @retries(5)
    def on_prepare(self):
        with sdk2.helpers.ProgressMeter("Checkout"):
            self.repo = self.Utils.vcs_helper.clone_with_task(self)
        with sdk2.helpers.ProgressMeter("Prepare signing keyring file"):
            self._prepare_keyring_file()
        with sdk2.helpers.ProgressMeter("Extract input resource"):
            self._prepare_input_resource()

    def _work_dir(self, *path):
        return str(self.path("wd", *path))

    @property
    def logger(self):
        try:
            return self._logger
        except AttributeError:
            self._logger = logging.getLogger("scenario")
        return self._logger

    @property
    def env(self):
        try:
            return self._env
        except AttributeError:
            self._env = self.Utils.shell_executor.default_env
            GradleParameters.prepare_env(self, self._env)
            if 'JAVA_HOME' not in self._env:
                self._env['JAVA_HOME'] = '/usr/local/java8'

            tmpdir = self.ramdrive.path if self.Parameters.use_tmpfs else None
            ShellExecutor.prepare_java_options(self._env, tmpdir)
        return self._env

    @property
    def execute_shell_timeout(self):
        exec_time = self.server.task.current.read()["execution"]["current"]
        remaining_time = self.Parameters.kill_timeout - exec_time
        self.logger.info("exec_time %d sec, remains %d sec, shell timeout: %d sec", exec_time, remaining_time, remaining_time)
        return remaining_time

    def log_path_custom(self, *path):
        """
        Build path relative to custom logs - see log_path
        :param path: path components
        :rtype: pathlib2.Path
        """
        p = self.path("artifacts", *path)
        if not p.exists():
            os.makedirs(str(p))
        return p

    @sdk2.header()
    @custom_report_logger
    def report(self):
        reports = self.Parameters.output_resource_reports or {}
        if self.Context.artifacts_resource_id is not None:
            template_context = {"resource_link": sdk2.Resource[self.Context.artifacts_resource_id].http_proxy,
                                "reports": reports}
            return jinja2.Environment(loader=jinja2.FileSystemLoader(os.path.dirname(os.path.abspath(__file__))),
                                      extensions=['jinja2.ext.do']).get_template("header.html").render(template_context)
        else:
            return None

    def create_teamcity_artifacts(self):
        if not self.Parameters.teamcity.teamcity_artifact_paths:
            return
        self.Utils.teamcity_publisher.create_teamcity_artifacts(self, self._work_dir(),
                                                                self.Parameters.teamcity.teamcity_artifact_paths,
                                                                self._get_teamcity_messages_path())

    def _get_teamcity_messages_path(self):
        return str(self.path("teamcity_messages.log"))

    def _write_error_to_teamcity(self, message):
        with open(self._get_teamcity_messages_path(), "a") as fout:
            fout.write("##teamcity[buildStatus text='{0}']".format(
                message.replace("|", "||").replace("'", "|'").replace("[", "|[").replace("]", "|]").replace("\n", "|n")))
