# coding=utf-8
import pathlib2

import sandbox.sdk2.helpers.process
from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.metrika.mobile.sdk.utils.RunTestsBase import RunTestsBase
from sandbox.projects.mobile_apps.utils.jdk_env import JdkEnvironment
from sandbox.projects.common.decorators import retries
from sandbox.projects.metrika.mobile.sdk.helpers.VcsHelper import VcsHelper
from sandbox.projects.metrika.mobile.sdk.parameters.VcsParameters import VcsParameters
from sandbox.projects.metrika.utils import custom_report_logger

DEFAULT_GRADLE_PARAMETERS = {}

DEFAULT_BRANCH = "dev"
TMP_UNITY_PLUGIN_PATH = str(pathlib2.Path.home().joinpath('mobileadssdk-unity'))


class BuildAdLibUnityPlugin(RunTestsBase):
    """
    Build adlib unity plugin build
    """

    class Utils:
        vcs_helper = VcsHelper()

    class Requirements(RunTestsBase.Requirements):
        client_tags = Tag.CUSTOM_OSX_JAVA & Tag.USER_UNITY & ~Tag.USER_BROKE

    class Parameters(RunTestsBase.Parameters):
        vcs = VcsParameters

        plugin_path = sdk2.parameters.String("Unity plugin path",
                                             default="adv/pcode/mobileadssdk/sdk/unity/plugin",
                                             required=True,
                                             description="Path to unity plugin")
        gradle_parameters = sdk2.parameters.Dict("Gradle Parameters",
                                                 default=DEFAULT_GRADLE_PARAMETERS)
        jdk_version = sdk2.parameters.String("JDK version",
                                             default="8u202",
                                             required=True,
                                             description="Available versions: https://nda.ya.ru/t/4Rh1g6lN3nA8qp")

    @retries(5)
    def on_prepare(self):
        with sdk2.helpers.ProgressMeter('JDK {} prepare'.format(self.Parameters.jdk_version)):
            JdkEnvironment(self.Parameters.jdk_version).prepare()
        with sandbox.sdk2.helpers.ProgressMeter("Checkout"):
            self.repo = self.Utils.vcs_helper.clone_with_task(self)

    def on_execute(self):
        with self.repo:
            with sandbox.sdk2.helpers.ProgressMeter("Precheck"):
                self._precheck()
            try:
                with sandbox.sdk2.helpers.ProgressMeter("Move unity plugin out of arc"):
                    self._move_unity_plugin()
                with sandbox.sdk2.helpers.ProgressMeter("Build main packages"):
                    self._build_unity_plugin()
                with sandbox.sdk2.helpers.ProgressMeter("Build mediation packages"):
                    self._build_mediation_packages()
                with sandbox.sdk2.helpers.ProgressMeter("Copy output files"):
                    self._copy_output_files()
                with sandbox.sdk2.helpers.ProgressMeter("Copy mediation output files"):
                    self._copy_mediation_output_files()
                with sandbox.sdk2.helpers.ProgressMeter("Remove tmp unity plugin dir"):
                    self._remove_tmp_unity_plugin()
            except Exception:
                self.logger.error("Exception in scenario", exc_info=True)
                raise
            finally:
                self._create_custom_logs_resource()

    def _precheck(self):
        # fail task if there was failed tests
        # small diagnostic and basic check
        self._execute_shell_and_check('printenv', env=self.env)
        self._execute_shell_and_check('whoami')
        self._execute_shell_and_check(["git", "--version"])
        self._execute_shell_and_check(["java", "-version"], env=self.env)
        self._execute_shell_and_check(["cat", "/etc/resolv.conf"])

    def _move_unity_plugin(self):
        self._execute_shell_and_check(["mkdir", TMP_UNITY_PLUGIN_PATH], env=self.env)
        self._execute_shell_and_check(["cp", "-R", "./", "%s/" % TMP_UNITY_PLUGIN_PATH],
                                      cwd=self._work_dir(self.Parameters.plugin_path),
                                      env=self.env)

    def _remove_tmp_unity_plugin(self):
        self._execute_shell_and_check(["rm", "-rf", TMP_UNITY_PLUGIN_PATH])

    def _build_unity_plugin(self):
        self._execute_gradle(TMP_UNITY_PLUGIN_PATH,
                             ["--info", ":makeUnityPlugin"],
                             gradle_progerties=self._get_gradle_unity_plugin_properties())

    def _build_mediation_packages(self):
        self._execute_gradle(TMP_UNITY_PLUGIN_PATH,
                             ["--info", ":buildMediationPackages"],
                             gradle_progerties=self._get_gradle_unity_plugin_properties())

    def _copy_output_files(self):
        self._execute_gradle(TMP_UNITY_PLUGIN_PATH,
                             ["--info", ":copyOutputFiles"],
                             gradle_progerties=self._get_gradle_unity_plugin_properties())

    def _copy_mediation_output_files(self):
        self._execute_gradle(TMP_UNITY_PLUGIN_PATH,
                             ["--info", ":copyMediationOutputFiles"],
                             gradle_progerties=self._get_gradle_unity_plugin_properties())

    def _get_gradle_unity_plugin_properties(self):
        unity_plugin_properties = {
            "plugin.output.dir": self._get_common_output_dir(),
        }
        unity_plugin_properties.update({k: v for k, v in self.Parameters.gradle_parameters.iteritems() if v})

        return unity_plugin_properties

    def _work_dir(self, *path):
        return str(self.path("wd", *path))

    def _get_common_output_dir(self):
        return str(self.log_path_custom("output").absolute())

    @sdk2.header()
    @custom_report_logger
    def header(self):
        if self.Context.artifacts_resource_id is not None:
            proxy = sdk2.Resource[self.Context.artifacts_resource_id].http_proxy
            footer = []
            append_artifacts_footer(footer, proxy)
            return footer
        else:
            return None


def append_artifacts_footer(footer, resource_link):
    if resource_link:
        footer.append({
            "content": {
                "<h4>Artifacts</h4>": {
                    "header": [
                        {"key": "unity_packages", "title": ""},
                        {"key": "unity_log", "title": ""},
                    ],
                    "body": {
                        "unity_packages": ["<a href='{proxy!s}/output/' target='_blank'>Unity Packages</a>".format(proxy=resource_link)],
                        "unity_log": ["<a href='{proxy!s}/output/unity.log' target='_blank'>Unity Log</a>".format(proxy=resource_link)],
                    }
                }
            },
            "helperName": ""
        })
