import os

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects.sandbox import sandbox_lxc_image


def load_file(filename):
    filename = os.path.join(os.path.dirname(os.path.realpath(__file__)), filename)
    with open(filename) as fh:
        return fh.read()


def get_default_emulators():
    default = []
    all_api_levels = {17, 19, 22, 23, 24, 25, 26, 27, 28, 29}
    for api_level in all_api_levels:
        emulator_description = {
            "api_level": api_level,
            "platform": "x86",
            "api": "google_apis",
            "avdmanager_create_args": ""
        }
        default.append(emulator_description)
    for api_level in all_api_levels - {17}:
        emulator_description = {
            "api_level": api_level,
            "platform": "x86",
            "api": "default",
            "avdmanager_create_args": ""
        }
        default.append(emulator_description)
    for api_level in all_api_levels - {17, 19, 27}:
        emulator_description = {
            "api_level": api_level,
            "platform": "x86_64",
            "api": "google_apis",
            "avdmanager_create_args": ""
        }
        default.append(emulator_description)
    for api_level in all_api_levels - {17, 19}:
        emulator_description = {
            "api_level": api_level,
            "platform": "x86_64",
            "api": "default",
            "avdmanager_create_args": ""
        }
        default.append(emulator_description)
    return default


class CreateAndroidEmulatorsLxcContainer(sdk2.Task):
    """
    Run task SANDBOX_LXC_IMAGE with particular custom script and build emulator containers for each supported
    API level
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        version = sdk2.parameters.String("Version", required=True, default="stable")
        emulators = sdk2.parameters.JSON("Emulators description", required=True, default=get_default_emulators())
        script_template = sdk2.parameters.String("Script template",
                                                 required=False,
                                                 multiline=True,
                                                 description="Default template will be used if empty")

    class Context(sdk2.Task.Context):
        children = []

    def _get_task(self, emulator_description):
        script_template = self.Parameters.script_template or load_file("prepare_container_template.sh")
        script = script_template.format(**emulator_description)
        description = "Create LXC container for API level {api_level} {api} {platform}".format(**emulator_description)

        image_attrs = {
            "frankenstein_android_emulator": True,
            "emulator_description": "{api_level};{api};{platform}".format(**emulator_description),
            "version": self.Parameters.version
        }

        params = {
            sandbox_lxc_image.SandboxLxcImage.Parameters.install_common.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.ubuntu_release.name: sandbox_lxc_image.UbuntuRelease.TRUSTY,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_image.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.test_result_lxc.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_attrs.name: image_attrs,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_script.name: script,
            sandbox_lxc_image.SandboxLxcImage.Parameters.resource_description.name:
                "Frankenstein emulator with description {}".format(emulator_description)
        }

        return sandbox_lxc_image.SandboxLxcImage(
            self,
            description=description,
            priority=min(
                self.Parameters.priority,
                ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)  # default API limitation
            ),
            **params
        )

    def on_execute(self):
        with self.memoize_stage.create_images(commit_on_entrance=False):
            for emulator_description in self.Parameters.emulators:
                task = self._get_task(emulator_description)
                task.enqueue()
                self.Context.children.append(task.id)

            raise sdk2.WaitTask(self.Context.children, tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK))

        with self.memoize_stage.check_children(commit_on_entrance=False):
            if not all(task.status in ctt.Status.Group.SUCCEED for task in self.find(id=self.Context.children)):
                raise common.errors.TaskFailure("Not all LXC containers built (see subtasks for details)")
