import os

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox import common
from sandbox import sdk2
from sandbox.projects.sandbox import sandbox_lxc_image


def load_file(filename):
    filename = os.path.join(os.path.dirname(os.path.realpath(__file__)), filename)
    with open(filename) as fh:
        return fh.read()


def get_default_parameters():
    return {
        "ndk_version": "21.0.6113669",
        "cmake_version": "3.10.2.4988404",
    }


class CreateAndroidGradleRunnerContainer(sdk2.Task):
    """
    Create container to run gradle tasks for android project
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        version = sdk2.parameters.String("version", required=True, default="stable")
        parameters = sdk2.parameters.JSON("Parameters for template", required=True, default=get_default_parameters())
        script_template = sdk2.parameters.String("Script template",
                                                 required=False,
                                                 multiline=True,
                                                 description="Default template will be used if empty")

    class Context(sdk2.Task.Context):
        children = []

    def _get_task(self):
        script_template = self.Parameters.script_template or load_file("prepare_container_template.sh")
        script = script_template.format(**self.Parameters.parameters)
        description = "Create container to run gradle tasks for android project"
        image_attrs = {
            "android_gradle_runner": True,
            "version": self.Parameters.version
        }
        params = {
            sandbox_lxc_image.SandboxLxcImage.Parameters.install_common.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.ubuntu_release.name: sandbox_lxc_image.UbuntuRelease.TRUSTY,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_image.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.test_result_lxc.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_attrs.name: image_attrs,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_script.name: script,
            sandbox_lxc_image.SandboxLxcImage.Parameters.resource_description.name: "build container",
        }
        return sandbox_lxc_image.SandboxLxcImage(
            self,
            description=description,
            priority=min(
                self.Parameters.priority,
                ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)  # default API limitation
            ),
            **params
        )

    def on_execute(self):
        with self.memoize_stage.create_images(commit_on_entrance=False):
            task = self._get_task()
            task.enqueue()
            self.Context.children.append(task.id)
            raise sdk2.WaitTask(self.Context.children, tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK))
        with self.memoize_stage.check_children(commit_on_entrance=False):
            if not all(task.status in ctt.Status.Group.SUCCEED for task in self.find(id=self.Context.children)):
                raise common.errors.TaskFailure("Not all LXC containers built (see subtasks for details)")
