import os
import tempfile

from sandbox import sdk2
from sandbox.common.config import Registry
from sandbox.common.types.client import Tag
from sandbox.projects.metrika.mobile.sdk.utils import Rvm
from sandbox.projects.metrika.mobile.sdk.utils import RvmEnvironment


def load_file(filename):
    filename = os.path.join(os.path.dirname(os.path.realpath(__file__)), filename)
    with open(filename) as fh:
        return fh.read()


class CreateRvmEnvironmentResource(sdk2.Task):
    """
        Creates resource with RVM environment of given version
    """

    name = "CREATE_RVM_ENVIRONMENT_RESOURCE"

    class Requirements(sdk2.Task.Requirements):
        client_tags = Tag.CUSTOM_OSX_JAVA & ~Tag.USER_BROKE

    class Parameters(sdk2.Task.Parameters):
        version = sdk2.parameters.String("version", required=True, default=RvmEnvironment.STABLE)

    def on_execute(self):
        destination_dir = str(self.path("install"))
        with sdk2.helpers.ProgressMeter("Install RVM"):
            source_dir = str(self.path("source"))
            with tempfile.NamedTemporaryFile() as script_file:
                script_file.write(load_file("install_template.sh").format(
                    version=self.Parameters.version, source=source_dir, destination=destination_dir))
                script_file.flush()
                with sdk2.helpers.ProcessLog(self, logger="install") as log:
                    sdk2.helpers.subprocess.call(['/bin/bash', script_file.name],
                                                 stdout=log.stdout,
                                                 stderr=sdk2.helpers.process.subprocess.STDOUT)

        with sdk2.helpers.ProgressMeter("Create Resource"):
            tarball_path = str(self.path('rvm.tar.gz'))
            with sdk2.helpers.ProcessLog(self, logger="compress") as log:
                sdk2.helpers.subprocess.call(['tar', '-zcvf', tarball_path,
                                              '--directory', destination_dir, '.'],
                                             stdout=log.stdout,
                                             stderr=sdk2.helpers.process.subprocess.STDOUT)
            resource = Rvm(self, "RVM {}".format(self.Parameters.version), tarball_path,
                           Registry().this.system.family,
                           version=self.Parameters.version,
                           platform="any",
                           ttl="inf")
            sdk2.ResourceData(resource).ready()
