# coding=utf-8
import logging
import tarfile
import os

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.common.types.resource import RestartPolicy
from sandbox.projects.metrika.mobile.sdk.helpers.DistributionCertificateHelper import DistributionCertificateHelper
from sandbox.projects.metrika.mobile.sdk.helpers.GitHelper import GitHelper
from sandbox.projects.metrika.mobile.sdk.helpers.MacIosHelper import MacIosHelper
from sandbox.projects.metrika.mobile.sdk.helpers.ProvisioningHelper import ProvisioningHelper
from sandbox.projects.metrika.mobile.sdk.helpers.ShellExecutor import ShellExecutor
from sandbox.projects.metrika.mobile.sdk.helpers.TeamCityArtifactPublisher import TeamCityArtifactPublisher
from sandbox.projects.metrika.mobile.sdk.helpers.VcsHelper import VcsHelper
from sandbox.projects.metrika.mobile.sdk.parameters.IosParameters import IosRunnerParameters
from sandbox.projects.metrika.mobile.sdk.parameters.ResourcePublishParameters import ResourcePublishParameters
from sandbox.projects.metrika.mobile.sdk.parameters.VcsParameters import VcsParameters
from sandbox.sdk2 import Resource


class FastlaneResource(Resource):
    """
    Task logs
    """
    name = "FASTLANE_RESOURCE"
    restart_policy = RestartPolicy.IGNORE


class FastlaneRunner(sdk2.Task):
    """
    Run fastlane lanes
    """
    privileged = True

    class Utils:
        vcs_helper = VcsHelper()
        git_helper = GitHelper()
        shell_executor = ShellExecutor()
        provisioning_helper = ProvisioningHelper(git_helper)
        distribution_certificate_helper = DistributionCertificateHelper(shell_executor)
        teamcity_publisher = TeamCityArtifactPublisher()
        ios_helper = MacIosHelper()

    class Requirements(sdk2.Requirements):
        client_tags = Tag.CUSTOM_OSX_JAVA & ~Tag.USER_BROKE
        dns = DnsType.DNS64
        disk_space = 8192

    class Parameters(sdk2.Parameters):
        ios_parameters = IosRunnerParameters
        vcs = VcsParameters
        ssh_username = sdk2.parameters.String("Ssh username",
                                              description="Какому пользователю принадлежит указанный ssh ключ "
                                                          "(необходимо для работы с yav)",
                                              default="")

        with sdk2.parameters.Group("Ruby parameters") as ruby_group:
            rvm_version = sdk2.parameters.String("RVM version (deprecated)",
                                                 description="Use rvm_plus_ruby_version instead")
            ruby_version = sdk2.parameters.String("Ruby version (deprecated)",
                                                  description="Use rvm_plus_ruby_version instead")
            bundler_version = sdk2.parameters.String("Bundler version")

        with sdk2.parameters.Group("Fastlane parameters") as fastlane_group:
            project_dir = sdk2.parameters.String("Project dir",
                                                 description="Откуда будут выполняться команды",
                                                 default=".")
            fastlane_platform = sdk2.parameters.String("Platform",
                                                       description="Для какой платформы будут выполняться lane'ы")
            fastlane_lanes = sdk2.parameters.List("Lanes",
                                                  description="Lane'ы, которые будут выполнены",
                                                  default=[])
            fastlane_parameters = sdk2.parameters.Dict("Параметры",
                                                       description="Параметры, которые передаются в fastlane",
                                                       default={})
            secret_fastlane_parameters = sdk2.parameters.Dict("Секретные параметры",
                                                              description="Параметры, которые будут получены из vault "
                                                                          "и переданы в fastlane, значения необходимо "
                                                                          "задавать в формате owner:name",
                                                              default={})
            env = sdk2.parameters.Dict("Параметры окружения",
                                       default={})

        with sdk2.parameters.Group("Security configuration") as security:
            import_provisioning = sdk2.parameters.Bool("Import provisioning", default=False)
            import_certificate = sdk2.parameters.Bool("Import certificate", default=False)
            certificate = sdk2.parameters.Vault("Сертификат для подписи приложения",
                                                description="Сертификат, который берётся из vault, "
                                                            "значение необходимо задавать в формате owner:name")

        resource = ResourcePublishParameters
        teamcity_artifact_paths = sdk2.parameters.Dict("Artifact paths",
                                                       description="Пути к файлам или папками, которые необходимо "
                                                                   "опубликовать в качестве артифактов сборки "
                                                                   "на Team City")

    def on_save(self):
        self.Utils.ios_helper.on_save(self)

    def on_prepare(self):
        self.Utils.ios_helper.on_prepare(self)
        with sdk2.helpers.ProgressMeter("Checkout"):
            self.repo = self.Utils.vcs_helper.clone_with_task(self)
        if self.Parameters.import_provisioning:
            with sdk2.helpers.ProgressMeter("Clone provisioning"):
                self._clone_provisioning()
            with sdk2.helpers.ProgressMeter("Prepare provisioning"):
                self._prepare_provisioning()
        if self.Parameters.import_certificate:
            with sdk2.helpers.ProgressMeter("Prepare certificate"):
                self._prepare_certificate()

    def on_execute(self):
        with self.repo:
            try:
                if not self.Parameters.rvm_plus_ruby_version:
                    with sdk2.helpers.ProgressMeter("Prepare ruby {}".format(self.Parameters.ruby_version)):
                        self._prepare_ruby_environment()
                with sdk2.helpers.ProgressMeter("Bundle install"):
                    self._bundle_install()
                with sdk2.helpers.ProgressMeter("Run lanes"):
                    self._run_fastlane_lanes()
            except Exception:
                self.logger.error("Exception in scenario", exc_info=True)
                raise
            finally:
                with sdk2.helpers.ProgressMeter("Save built resource"):
                    self._save_to_resource()
                with sdk2.helpers.ProgressMeter("Pass artifacts to teamcity"):
                    self._create_teamcity_artifacts()
                with sdk2.helpers.ProgressMeter("Remove files"):
                    self._remove_files()

    def _root_dir(self, *path):
        return str(self.path(*path))

    def _work_dir(self, *path):
        return self._root_dir("wd", *path)

    def _project_dir(self, *path):
        return self._work_dir(self.Parameters.project_dir, *path)

    def _clone_provisioning(self):
        self.Utils.provisioning_helper.clone_provisioning(self._root_dir("provision"), self.Parameters.vcs.ssh_key)

    def _prepare_provisioning(self):
        self.Utils.provisioning_helper.prepare_provisioning(self.logger, self._root_dir("provision"))

    def _prepare_certificate(self):
        password = sdk2.Vault.data('METRIKA', 'YMAKE_DEV_MAC_SANDBOX_USER_PASSWORD')
        self.Utils.distribution_certificate_helper.import_certificate(
            certificate=self.Parameters.certificate,
            password=password,
            work_dir=self._project_dir()
        )

    def _prepare_ruby_environment(self):
        """
        Prepare ruby envirnoment
        :return:
        """
        self.logger.info("Ruby working directory is: {0}".format(self._project_dir()))

        ruby_install_args = ['rvm', 'install', 'ruby-{0}'.format(self.Parameters.ruby_version)]
        self.Utils.shell_executor.execute_shell_and_check(ruby_install_args, self._project_dir(), env=self.env,
                                                          timeout=self.execute_shell_timeout, logger_name='rvm_install_{0}'.format(self.Parameters.ruby_version))

        rvm_clean_args = ['rvm', 'gemset', 'empty', '--force']
        self._execute_in_ruby_env(rvm_clean_args, self._project_dir(), logger_name='gemset_empty')

    def _bundle_install(self):
        gem_install_args = ['gem', 'install', 'bundler', '--no-document']
        if self.Parameters.bundler_version:
            gem_install_args += ['-v', self.Parameters.bundler_version]
        self._execute_in_ruby_env(gem_install_args, self._project_dir(), logger_name='gem_install')

        bundle_install_args = ['bundle', 'install', '--full-index']
        self._execute_in_ruby_env(bundle_install_args, self._project_dir(), logger_name='bundle_install')

    def _run_fastlane_lanes(self):
        """
        Running list of fastlane lanes
        """

        def run_lanes():
            platform = self.Parameters.fastlane_platform
            properties = self._get_fastlane_properties()
            if self.Parameters.fastlane_lanes:
                for lane in self.Parameters.fastlane_lanes:
                    self.logger.info('Run lane: ' + lane)
                    self._execute_fastlane(self._project_dir(), lane, platform, properties,
                                           logger_name='run_lane_{}_{}'.format(platform, lane))

        ssh_key = self.Parameters.vcs.ssh_key
        if ssh_key:
            with sdk2.ssh.Key(self, ssh_key.owner, ssh_key.name):
                run_lanes()
        else:
            run_lanes()

    def _save_to_resource(self):
        """
        Save directory to resource
        """
        if self.Parameters.resource.resource_path:
            res_file = tarfile.open(self._root_dir("fastlane_resource.tar.gz"), "w:gz")
            res_file.add(
                self._work_dir(self.Parameters.resource.resource_path),
                arcname=os.path.basename(self._work_dir(self.Parameters.resource.resource_path))
            )
            res_file.close()
            self.Parameters.composed_resource = FastlaneResource(self, "Built Resource", res_file.name)
            sdk2.ResourceData(self.Parameters.resource.composed_resource).ready()

    def _create_teamcity_artifacts(self):
        """
        Create artifact for teamcity
        """
        if self.Parameters.teamcity_artifact_paths:
            self.Utils.teamcity_publisher.create_teamcity_artifacts(
                self,
                self._work_dir(),
                self.Parameters.teamcity_artifact_paths,
                str(self.path("teamcity_messages.log"))
            )

    def _remove_files(self):
        self.Utils.provisioning_helper.remove_files()

    def _get_fastlane_properties(self):
        """
        :return: map of fastlane properties
        """
        fastlane_properties = {}
        fastlane_properties.update({k: v for k, v in self.Parameters.fastlane_parameters.iteritems() if v})
        fastlane_properties.update({k: sdk2.Vault.data(*v.split(':', 1))
                                    for k, v in self.Parameters.secret_fastlane_parameters.iteritems() if v})

        return fastlane_properties

    def _execute_fastlane(self, cwd, lane, platform=None, properties=None, logger_name=None):
        """
        :param cwd: path to fastlane execution dir
        :param lane: lanes that will be executed...
        :param properties: lane parameters
        :param logger_name: logger name
        :return:
        """
        args = ['bundle', 'exec', 'fastlane']
        if platform:
            args += [platform]
        args += [lane]
        if properties:
            args += ["{0}:{1}".format(k, v) for (k, v) in properties.iteritems()]

        self.logger.info('Arguments:{0}'.format(args))
        return self._execute_in_ruby_env(args, cwd, logger_name=logger_name)

    def _execute_in_ruby_env(self, args, cwd=None, logger_name=None):
        """
        execute command in ruby environment
        :param args: command for execution
        :param cwd: path for working directory
        :param logger_name: logger name
        :return: success or not
        """
        env_args = ['rvm', 'in', cwd, 'do']
        return self.Utils.shell_executor.execute_shell_and_check(env_args + args, cwd, self.env,
                                                                 self.execute_shell_timeout, logger_name=logger_name)

    @property
    def execute_shell_timeout(self):
        """
        :return: shell execution timeout in seconds
        """
        exec_time = self.server.task.current.read()["execution"]["current"]
        remaining_time = self.Parameters.kill_timeout - exec_time
        timeout = remaining_time
        self.logger.info("exec_time %d sec, remains %d sec, shell timeout: %d sec", exec_time, remaining_time, timeout)
        return timeout

    @property
    def env(self):
        """
        :return: Map of environment variables
        """
        env = self.Utils.shell_executor.default_env
        self.Utils.ios_helper.prepare_env(self, env)
        if self.Parameters.ssh_username:
            env['YAV_RSA_LOGIN'] = self.Parameters.ssh_username
        for key, value in self.Parameters.env.iteritems():
            env[key] = value
        return env

    @property
    def logger(self):
        try:
            return self._logger
        except AttributeError:
            self._logger = logging.getLogger("scenario")
        return self._logger
