# coding=utf-8
import copy
import jinja2
import os
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.common.errors import TaskError, TaskFailure
from sandbox.common.types.misc import DnsType
from sandbox.common.utils import get_task_link
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.common.utils import colored_status
from sandbox.projects.metrika.mobile.sdk.generics.generic_launcher.utils import parse_config_from_file, parse_config_from_str
from sandbox.projects.metrika.mobile.sdk.helpers.ArcHelper import ArcHelper
from sandbox.projects.metrika.mobile.sdk.helpers.EmptyContext import EmptyContext
from sandbox.projects.metrika.utils import custom_report_logger

WORKING_DIR = "wd"


class GenericLauncher(sdk2.Task):
    """
    Запускает таски из конфига
    """

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64
        disk_space = 1024
        cores = 1
        ram = 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 10 * 60
        with sdk2.parameters.Group("VCS") as vcs_group:
            vcs_url = sdk2.parameters.String("Адрес репозитория",
                                             description="Поддерживается только 'arcadia'",
                                             default="arcadia")
            branch = sdk2.parameters.String("Ветка или хэш коммита",
                                            description="Ветка или хэш коммита репозитория")
            commit = sdk2.parameters.String("Хэш коммита",
                                            description="Заполняется автоматически teamcity")
            ssh_key_yav = sdk2.parameters.YavSecret("SSH-ключ yav",
                                                    description="SSH-ключ, который берётся из yav и будет "
                                                                "использоваться для клонирования репозитория")
        with sdk2.parameters.Group("Configs") as configs_group:
            yaml_config_path = sdk2.parameters.String("Путь к yaml конфигу",
                                                      description="Путь к конфигу относительно корня репозитория")
            yaml_config = sdk2.parameters.String("yaml конфиг",
                                                 description="Конфиг в формате https://nda.ya.ru/t/VjhPUwSa3tsWZ9",
                                                 multiline=True)

    class Utils:
        arc_helper = ArcHelper()

    class Context(sdk2.Task.Context):
        commit_hash = None
        children = {}

    def on_prepare(self):
        if self.Parameters.vcs_url and (self.Parameters.branch or self.Parameters.commit):
            if self.Parameters.vcs_url != "arcadia":
                raise TaskError("vcs url {} not supported".format(self.Parameters.vcs_url))
            with sdk2.helpers.ProgressMeter("Clone repository"):
                self.repo = self.Utils.arc_helper.clone_repository(
                    working_directory=self._work_dir(),
                    branch=self.Parameters.branch,
                    commit=self.Parameters.commit,
                )
                self.Context.commit_hash = self.Parameters.commit or Arc().info(self._work_dir())["hash"]
        else:
            self.repo = EmptyContext()

    def on_execute(self):
        with self.repo:
            with sdk2.helpers.ProgressMeter("Run tasks"):
                self._run_and_wait_tasks()

    def _work_dir(self, *path):
        return str(self.path(WORKING_DIR, *path))

    def _run_and_wait_tasks(self):
        with self.memoize_stage.launch_tasks(commit_on_entrance=False):
            for task_id, task_info in self._get_tasks().iteritems():
                task_info["task"].enqueue()
                self.Context.children.update({task_info["task"].id: task_info["name"]})

            raise sdk2.WaitTask(self.Context.children.keys(),
                                tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK))

        with self.memoize_stage.check_children(commit_on_entrance=False):
            self._check_children()

    def _check_children(self):
        statuses = [(name, sdk2.Task[task_id].status) for (task_id, name) in self.Context.children.items()]
        failed_statuses = filter(lambda status: status[1] not in ctt.Status.Group.SUCCEED, statuses)

        if failed_statuses:
            raise TaskFailure("\n" + "\n".join(["{0}: {1}".format(name, status) for (name, status) in failed_statuses]))

    def _get_tasks(self):
        task_configs = self._get_task_configs()
        return {task_id: {"name": task_config.get("name", task_id), "task": self._get_task(task_config)}
                for task_id, task_config in task_configs.iteritems()}

    def _get_task(self, task_config):
        params = copy.deepcopy(task_config["info"])
        params.update(task_config["parameters"])
        self._append_vcs_params(params)
        return sdk2.Task[task_config["type"]](self, __requirements__=task_config["requirements"], **params)

    def _append_vcs_params(self, params):
        vcs_params = {
            GenericLauncher.Parameters.vcs_url.name: self.Parameters.vcs_url,
            GenericLauncher.Parameters.branch.name: self.Parameters.branch,
            GenericLauncher.Parameters.commit.name: self.Context.commit_hash or self.Parameters.commit,
            GenericLauncher.Parameters.ssh_key_yav.name: self.Parameters.ssh_key_yav,
        }
        for name, value in vcs_params.iteritems():
            # TODO: check param in task
            params.setdefault(name, value)

    def _get_task_configs(self):
        tasks = {}
        if self.Parameters.yaml_config:
            tasks = parse_config_from_str(self.Parameters.yaml_config, self._work_dir()).get("tasks", {})
        if self.Parameters.yaml_config_path:
            tasks = parse_config_from_file(self._work_dir(self.Parameters.yaml_config_path), self._work_dir()).get("tasks", {})
        for _, task in tasks.iteritems():
            self._append_vcs_params(task)
        return tasks

    @sdk2.header()
    @custom_report_logger
    def report(self):
        if self.Parameters.yaml_config_path or self.Parameters.yaml_config:
            template_context = {"tasks": [
                {
                    "name": name,
                    "link": get_task_link(task_id),
                    "status": colored_status(sdk2.Task[task_id].status),
                }
                for (task_id, name) in sorted(self.Context.children.items())
            ]}
            return jinja2.Environment(loader=jinja2.FileSystemLoader(os.path.dirname(os.path.abspath(__file__))),
                                      extensions=["jinja2.ext.do"]).get_template("header.html").render(template_context)
