import yaml
import os
import copy
from sandbox.common.errors import TaskFailure


def deep_merge(a, b):
    """
    >>> a = { 'first' : { 'all_rows' : { 'pass' : 'dog', 'number' : '1' } } }
    >>> b = { 'first' : { 'all_rows' : { 'fail' : 'cat', 'number' : '5' } } }
    >>> deep_merge(a, b) == { 'first' : { 'all_rows' : { 'pass' : 'dog', 'fail' : 'cat', 'number' : '5' } } }
    """
    for key, value in b.iteritems():
        if isinstance(value, dict):
            node = a.setdefault(key, {})
            a[key] = deep_merge(node, value)
        else:
            a[key] = value
    return a


def resolve_path(path, relative_root, absolute_root):
    root, path = [absolute_root, path[1:]] if path.startswith("/") else [relative_root, path]
    if root is None:
        raise TaskFailure("Failed resolve path {}".format(path))
    return os.path.join(root, path)


def read_config(config_path):
    try:
        with open(config_path, "r") as f:
            return f.read()
    except Exception as e:
        raise TaskFailure(e, "Failed read config {}".format(config_path))


def yaml_get_array(config_str, params_name):
    is_need_block = False
    values = []
    for event in yaml.parse(config_str, Loader=yaml.SafeLoader):
        if isinstance(event, yaml.ScalarEvent) and event.value == params_name:
            is_need_block = True
        elif is_need_block and isinstance(event, yaml.ScalarEvent):
            values.append(event.value)
        elif is_need_block and isinstance(event, yaml.SequenceEndEvent):
            break
    return values


def read_include_files(config_str, config_dir, vcs_path):
    try:
        paths = yaml_get_array(config_str, "include_files")
        return [read_config(resolve_path(path, config_dir, vcs_path)) for path in paths]
    except Exception as e:
        raise TaskFailure(e, "Failed read include_files")


def read_import_tasks(config_str, config_dir, vcs_path):
    try:
        paths = yaml_get_array(config_str, "import_tasks")
        tasks = {}
        for path in paths:
            config = parse_config_from_file(resolve_path(path, config_dir, vcs_path), vcs_path)
            tasks.update(config.get("tasks", {}))
        return tasks
    except Exception as e:
        raise TaskFailure(e, "Failed read import_tasks")


def resolve_params(block):
    for key, value in copy.deepcopy(block).iteritems():
        if key.endswith("+"):
            real_key = key[:-1]
            if real_key in block:
                if not isinstance(value, type(block[real_key])):
                    raise TaskFailure("Can not merge '{}' with type '{}' to type '{}'".format(
                        real_key, type(value), type(block[real_key])
                    ))
                if isinstance(value, dict):
                    block[real_key].update(value)
                elif isinstance(value, list):
                    block[real_key].extend(value)
                else:
                    raise TaskFailure("Can not merge '{}' with type '{}'", real_key, type(value))
            else:
                block[real_key] = value
            del block[key]


def resolve_task(task, import_tasks):
    extends_task = import_tasks[task["extends"]] if "extends" in task else {}
    if len(extends_task) != 0 and extends_task["type"] != task["type"]:
        raise TaskFailure("Extends task should be type '{}'".format(task["type"]))
    for block_name in ["info", "requirements", "parameters"]:
        task[block_name] = deep_merge(copy.deepcopy(extends_task.get(block_name, {})), task.get(block_name, {}))
        try:
            resolve_params(task[block_name])
        except Exception as e:
            raise TaskFailure(e, "Failed resolve block '{}'".format(block_name))


def resolve_tasks(config, import_tasks):
    for task_id, task in config.get("tasks", {}).iteritems():
        try:
            resolve_task(task, import_tasks)
        except Exception as e:
            raise TaskFailure(e, "Failed resolve task '{}'".format(task_id))


def parse_config(config_str, config_path, vcs_path):
    try:
        config_dir = config_path and os.path.dirname(config_path)
        include_files = read_include_files(config_str, config_dir, vcs_path)
        import_tasks = read_import_tasks(config_str, config_dir, vcs_path)

        config_str = "\n".join(include_files + [config_str])
        config = yaml.safe_load(config_str)
        resolve_tasks(config, import_tasks)
        return config
    except Exception as e:
        raise TaskFailure(e, "Failed parse config {}".format(config_path or "from string"))


def parse_config_from_str(config_str, vcs_path):
    return parse_config(config_str, None, vcs_path)


def parse_config_from_file(config_path, vcs_path):
    return parse_config(read_config(config_path), config_path, vcs_path)
