# coding=utf-8
import re
from sandbox.common.errors import TaskError
from sandbox.common.types.client import Tag

DEFAULT_XCODE_PATH = '/Applications/Xcode11.3.app'

DEFAULT_SIMULATORS_LIST = [
    # TODO: https://st.yandex-team.ru/FRANKENSTEIN-978
    # {'version': '10.3.1', 'name': '10.3.1 (ymake)', 'checked': True},
    # {'version': '11.3', 'name': '11.3 (ymake)', 'checked': True},
    {'version': '12.1', 'name': '12.1 (ymake)', 'checked': True},
    {'version': '13.5', 'name': '13.5 (mobile)', 'checked': True},
    {'version': '14.4', 'name': '14.4 (mobile)', 'checked': False},
    {'version': '14.5', 'name': '14.5 (mobile)', 'checked': True},
    {'version': '15.0', 'name': '15.0 (mobile)', 'checked': True},
]


class AdvancedIosSimulatorConfigHelper:

    def __init__(self):
        self._default_simulators = {
            '12.1': {
                'mac_type': 'ymake',
                'ios_version': '12.1',
                'device_model': 'iPhone X[S|s]',
                'xcode_path': DEFAULT_XCODE_PATH,
            },
            '13.5': {
                'mac_type': 'mobile',
                'ios_version': '13.5',
                'device_model': 'iPhone 11',
                'xcode_version': '11.5',
                'osx_tags': Tag.OSX_BIG_SUR,
            },
            '14.4': {
                'mac_type': 'mobile',
                'ios_version': '14.4',
                'device_model': 'iPhone 11',
                'xcode_version': '12.4',
                'osx_tags': Tag.OSX_BIG_SUR,
            },
            '14.5': {
                'mac_type': 'mobile',
                'ios_version': '14.5',
                'device_model': 'iPhone 11',
                'xcode_version': '12.5',
                'osx_tags': Tag.OSX_BIG_SUR,
            },
            '15.0': {
                'mac_type': 'mobile',
                'ios_version': '15.0',
                'device_model': 'iPhone 11',
                'xcode_version': '13.0',
                'osx_tags': Tag.OSX_BIG_SUR,
            },
        }
        self.ymake_matcher = re.compile('^(?:YMAKE_)?OS_(?P<ios_version>[^_]+)_(?P<device_model>[^_]+)(?:_(?P<xcode_path>.+))?$')
        self.mobile_matcher = re.compile('^MOBILE_XCODE_(?P<xcode_version>.+)_OS_(?P<ios_version>[^_]+)_(?P<device_model>[^_]+)(?:_TAGS_(?P<osx_tags>.+))?$')

    def get_simulator_config_by_ios_version(self, ios_version):
        return self._default_simulators[ios_version]

    def get_simulator_config(self, advanced_config):
        display_config = self.get_mobile_simulator_config(advanced_config) or self.get_ymake_simulator_config(advanced_config)
        if not display_config:
            raise TaskError('Failed parse simulator config: "{0}"'.format(advanced_config))
        return display_config

    def get_ymake_simulator_config(self, advanced_config):
        display_config = self.ymake_matcher.match(advanced_config)
        return {
            'mac_type': 'ymake',
            'ios_version': display_config.group('ios_version'),
            'device_model': display_config.group('device_model'),
            'xcode_path': display_config.group('xcode_path') or DEFAULT_XCODE_PATH,
        } if display_config else None

    def get_mobile_simulator_config(self, advanced_config):
        display_config = self.mobile_matcher.match(advanced_config)
        return {
            'mac_type': 'mobile',
            'xcode_version': display_config.group('xcode_version'),
            'ios_version': display_config.group('ios_version'),
            'device_model': display_config.group('device_model'),
            'osx_tags': display_config.group('osx_tags'),
        } if display_config else None
