import platform
import shutil

from sandbox import sdk2
from sandbox.projects.common.environments import SandboxGitLfsEnvironment
from sandbox.sdk2.vcs.git import Git, GitLfs


class GitHelper:

    def __init__(self):
        pass

    def clone_repository(self, working_directory, vcs_url, branch, ssh_key, need_git_lfs=False, use_latest_revision=False):
        def clone_repo():
            if need_git_lfs:
                SandboxGitLfsEnvironment('2.5.2').prepare()
            git = Git(vcs_url)
            try:
                git.update_cache_repo('HEAD')
                git.execute('clone', '--reference', git.cache_repo_dir, vcs_url, working_directory)
                git.execute('fetch', '--tags', 'origin', '+refs/heads/*:refs/heads/*',
                            '+refs/pull-requests/*:refs/pull-requests/*',
                            '+refs/pull/*:refs/pull/*',
                            '--update-head-ok', cwd=working_directory)
                if branch:
                    git.execute('checkout', branch, cwd=working_directory)
                if use_latest_revision:
                    git.execute('submodule', 'update', '--init', '--remote', '--recursive', cwd=working_directory)
                else:
                    git.execute('submodule', 'update', '--init', '--recursive', cwd=working_directory)
                git.execute('fetch', '--recurse-submodules=yes', '--tags', 'origin', '+refs/heads/*:refs/heads/*',
                            '+refs/pull-requests/*:refs/pull-requests/*', '--update-head-ok', cwd=working_directory)
            except:
                shutil.rmtree(working_directory)
                shutil.rmtree(git.cache_repo_dir)
                raise

        if ssh_key:
            ssh_key = ssh_key
            with sdk2.ssh.Key(
                key_owner=ssh_key.owner,
                key_name=ssh_key.name
            ):
                clone_repo()
        else:
            clone_repo()

    def new_clone_repository(self, working_directory, vcs_url, branch, commit, ssh_key, need_git_lfs=False):
        def clone_repo():
            if need_git_lfs:
                if platform.system() == 'Linux':
                    SandboxGitLfsEnvironment('2.5.2').prepare()
                git = GitLfs(vcs_url, filter_branches=False)
            else:
                git = Git(vcs_url, filter_branches=False)
            git.clone(working_directory, branch, commit=commit)

        if ssh_key:
            with sdk2.ssh.Key(
                key_owner=ssh_key.owner,
                key_name=ssh_key.name
            ):
                clone_repo()
        else:
            clone_repo()

    def clone_repository_with_open_ssh(self, working_directory, vcs_url, branch, commit, ssh_key, need_git_lfs=False):
        def clone_repo():
            if need_git_lfs:
                if platform.system() == 'Linux':
                    SandboxGitLfsEnvironment('2.5.2').prepare()
                git = GitLfs(vcs_url, filter_branches=False)
            else:
                git = Git(vcs_url, filter_branches=False)
            git.clone(working_directory, branch, commit=commit)

        if ssh_key:
            with sdk2.ssh.Key(
                private_part=ssh_key.data()["ssh"]
            ):
                clone_repo()
        else:
            clone_repo()

    def clone_with_task(self, task):
        """
        :param task: must have params from sandbox.projects.metrika.mobile.sdk.parameters.VcsParameters
        """
        if task.Parameters.ssh_key_yav:
            task.logger.info("Using yav git helper")
            self.clone_repository_with_open_ssh(
                working_directory=task._work_dir(),
                vcs_url=task.Parameters.vcs_url,
                branch=task.Parameters.branch,
                commit=task.Parameters.commit,
                ssh_key=task.Parameters.ssh_key_yav,
                need_git_lfs=task.Parameters.need_git_lfs)
        elif task.Parameters.use_new_git_helper:
            task.logger.info("Using new git helper")
            self.new_clone_repository(
                working_directory=task._work_dir(),
                vcs_url=task.Parameters.vcs_url,
                branch=task.Parameters.branch,
                commit=task.Parameters.commit,
                ssh_key=task.Parameters.ssh_key,
                need_git_lfs=task.Parameters.need_git_lfs)
        else:
            task.logger.info("Using old git helper")
            self.clone_repository(
                working_directory=task._work_dir(),
                vcs_url=task.Parameters.vcs_url,
                branch=task.Parameters.branch,
                ssh_key=task.Parameters.ssh_key,
                need_git_lfs=task.Parameters.need_git_lfs,
                use_latest_revision=task.Parameters.use_latest_revision
            )
