from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects.metrika.mobile.sdk.helpers.ShellExecutor import ShellExecutor
from sandbox.projects.metrika.mobile.sdk.parameters.IosParameters import MacType
from sandbox.projects.metrika.mobile.sdk.utils import RvmEnvironment
from sandbox.projects.mobile_apps.utils.jdk_env import JdkEnvironment
from sandbox.projects.mobile_apps.utils.rvm_plus_ruby_env import RvmPlusRubyEnvironment
from sandbox.projects.mobile_apps.utils import shellexecuter as mobile_apps_utils
from sandbox.sdk2.environments import Xcode


class MacIosHelper:

    def __init__(self):
        pass

    def on_save(self, task):
        """
        :param task: task with IosCommonParameters parameters
        """
        if task.Parameters.mac_type == MacType.YMAKE_DEV_MAC:
            task.Requirements.client_tags = Tag.CUSTOM_OSX_JAVA & ~Tag.USER_BROKE
        elif task.Parameters.mac_type == MacType.MOBILE_MAC:
            task.Requirements.client_tags = (Tag.MOBILE_MONOREPO | Tag.CUSTOM_MOBILE_MONOREPO) & Tag.USER_MONOREPO
            if task.Parameters.use_big_sur:
                task.Requirements.client_tags = task.Requirements.client_tags & Tag.OSX_BIG_SUR
            elif task.Parameters.use_catalina:
                task.Requirements.client_tags = task.Requirements.client_tags & Tag.OSX_CATALINA
            if task.Parameters.osx_tags:
                task.Requirements.client_tags = '{} & {}'.format(task.Requirements.client_tags,
                                                                 task.Parameters.osx_tags)

    def on_prepare(self, task):
        """
        :param task: task with IosRunnerParameters parameters
        """
        if task.Parameters.jdk_version:
            with sdk2.helpers.ProgressMeter('JDK {} prepare'.format(task.Parameters.jdk_version)):
                JdkEnvironment(task.Parameters.jdk_version).prepare()
        if task.Parameters.rvm_plus_ruby_version:
            with sdk2.helpers.ProgressMeter('Rvm+Ruby {} prepare'.format(task.Parameters.rvm_plus_ruby_version)):
                RvmPlusRubyEnvironment(
                    version=task.Parameters.rvm_plus_ruby_version,
                    platform=mobile_apps_utils.detect_platform(),
                ).prepare()
        if task.Parameters.mac_type == MacType.YMAKE_DEV_MAC:
            if not task.Parameters.rvm_plus_ruby_version:
                with sdk2.helpers.ProgressMeter('Rvm prepare'):
                    RvmEnvironment().prepare()
        elif task.Parameters.mac_type == MacType.MOBILE_MAC:
            if task.Parameters.xcode_version:
                with sdk2.helpers.ProgressMeter('Xcode {} prepare'.format(task.Parameters.xcode_version)):
                    Xcode(task.Parameters.xcode_version).prepare()
                with sdk2.helpers.ProgressMeter('Remove CoreSimulatorService'):
                    ShellExecutor().execute_shell(
                        ['launchctl', 'remove', 'com.apple.CoreSimulator.CoreSimulatorService'],
                        logger_name='remove_CoreSimulatorService'
                    )

    def get_xcode_path(self, task):
        """
        :param task: task with IosRunnerParameters parameters
        """
        xcode_path = None
        if task.Parameters.mac_type == MacType.YMAKE_DEV_MAC:
            xcode_path = task.Parameters.xcode_path
        elif task.Parameters.mac_type == MacType.MOBILE_MAC and task.Parameters.xcode_version:
            xcode_path = '/Applications/Xcode.app'
        return xcode_path

    def prepare_env(self, task, env):
        """
        :param task: task with IosCommonParameters parameters
        :param env: current environment
        """
        if task.Parameters.mac_type == MacType.YMAKE_DEV_MAC:
            if task.Parameters.rvm_plus_ruby_version:
                # ssl cert
                env['SSL_CERT_FILE'] = '/etc/ssl/cert.pem'
                # cert for slack
                env['WEBSOCKET_CLIENT_CA_BUNDLE'] = '/usr/local/etc/openssl/certs/YandexInternalRootCA.crt'
                env['SSL_CERT_DIR'] = '/etc/ssl/certs'
            if task.Parameters.xcode_path:
                env['DEVELOPER_DIR'] = '{}/Contents/Developer'.format(task.Parameters.xcode_path)
        elif task.Parameters.mac_type == MacType.MOBILE_MAC:
            if task.Parameters.rvm_plus_ruby_version:
                # certs for macos clients from arcadia/sandbox/deploy/layout/sandbox_macos_mobile/usr/local/etc/openssl
                # ssl cert
                env['SSL_CERT_FILE'] = '/usr/local/etc/openssl/allCAs.pem'
                # cert for slack
                env['WEBSOCKET_CLIENT_CA_BUNDLE'] = '/usr/local/etc/openssl/DigiCertGlobalRootCA.crt'
                env['SSL_CERT_DIR'] = '/usr/local/etc/openssl'
