import logging
import string
import time

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import subprocess


class ShellExecutor:

    def __init__(self):
        pass

    def execute_shell_and_check(self, args, cwd=None, env=None, timeout=None, logger_name=None):
        """
        if exit code is not zero throws exception
        :param args: command-line arguments, single string or sequence
        :param cwd: current dir
        :param env: environment variables
        :param timeout: command timeout
        :param logger_name: logger name
        :return: nothing
        """
        logger_name = logger_name or self._get_logger_name(args)
        exitcode = self.execute_shell(args, cwd=cwd, env=env, timeout=timeout, logger_name=logger_name)

        if exitcode:
            raise TaskFailure(
                "Command {!r}\nFAILED. Exit code: {!r}.\nSee log {!r}.err.log".format(args, exitcode, logger_name))

    def execute_shell(self, args, cwd=None, env=None, timeout=None, logger_name=None):
        """
        :param args: command-line arguments, single string or sequence
        :param cwd: current dir
        :param env: environment variables
        :param timeout: command timeout
        :param logger_name: logger name
        :return: exit code
        """
        env = env if env else self.default_env
        with sdk2.helpers.ProcessLog(logger=logger_name or self._get_logger_name(args),
                                     set_action=False,
                                     stdout_level=logging.DEBUG) as pl:
            pl.logger.setLevel(logging.DEBUG)
            pl.logger.info("\t\tCurrent environment is: %s", env)
            pl.logger.info("\t\tCMD: %s", args)
            pl.logger.info("\t\tWorking directory: %s", cwd)
            process = sdk2.helpers.subprocess.Popen(args, cwd=cwd, env=env,
                                                    stdout=pl.stdout,
                                                    stderr=pl.stderr)
            exit_code = process.wait(timeout=timeout)

            pl.logger.info("\t\tEXIT CODE: %s", exit_code)

            return exit_code

    def _get_logger_name(self, args):
        valid_chars = "-_.,:;() %s%s" % (string.ascii_letters, string.digits)
        logname = "{0}_{1}".format(time.time(), args[0])
        if len(args) > 1:
            logname = "{0}_{1}".format(logname, args[-1])

        return ''.join(c for c in logname if c in valid_chars)

    @property
    def default_env(self):
        env = dict(l.split('=', 1) for l in
                   subprocess.check_output(['/bin/bash', '-c', '. /etc/profile && printenv']).splitlines())
        env['LANG'] = 'en_US.UTF-8'
        return env

    @staticmethod
    def prepare_java_options(env, tmpdir=None):
        java_options = [
            env.get('_JAVA_OPTIONS', '-XX:MaxRAM=45G'),
            env.get('ADDITIONAL_JAVA_OPTIONS', ''),
        ]
        if tmpdir:
            java_options.append('-Djava.io.tmpdir={0}'.format(tmpdir))
        env['_JAVA_OPTIONS'] = ' '.join(java_options)
