import logging
import os
import shutil
import tarfile

from sandbox import sdk2
from sandbox.common import fs
from sandbox.projects.metrika.mobile.sdk.resources.GenericTestResource import GenericTestResource
from sandbox.projects.metrika.mobile.sdk.helpers.ShellExecutor import ShellExecutor
from sandbox.sdk2 import paths


logger = logging.getLogger('TarResourceHelper')


class TarResourceHelper:

    def __init__(self, shell_executor=ShellExecutor()):
        self.shell_executor = shell_executor

    def extract_input_resource(self, resource, path, task=None):
        fs.make_folder(path)
        if task is None:
            paths.add_write_permissions_for_path(str(path))
        else:
            # experiment for https://st.yandex-team.ru/FRANKENSTEIN-997
            add_write_permissions = getattr(task.Parameters, 'add_write_permissions', True)
            logging.info("Need add_write_permissions for path {}? {}".format(str(path), add_write_permissions))
            if add_write_permissions:
                paths.add_write_permissions_for_path(str(path))
        TarResourceHelper.unpack(str(sdk2.ResourceData(resource).path.as_posix()), str(path))

    def save_to_resource(self, task, description, current_dir, resource_dir_name, ttl=1, task_dir=None, backup=False):
        resource = self.create_resource(
            task=task,
            description=description,
            current_dir=current_dir,
            resource_dir_name=resource_dir_name,
            ttl=ttl,
            task_dir=task_dir,
            backup=backup
        )
        sdk2.ResourceData(resource).ready()
        return resource

    def create_resource(self, task, description, current_dir, resource_dir_name, ttl=1, task_dir=None, backup=False):
        tar_file_str = resource_dir_name + ".tar"
        TarResourceHelper.pack(tar_file_str, str(current_dir), str(current_dir / resource_dir_name))
        # experiment for https://st.yandex-team.ru/FRANKENSTEIN-997
        add_write_permissions = getattr(task.Parameters, 'add_write_permissions', True)
        logging.info("Need add_write_permissions for path {}? {}".format(
            str(current_dir / tar_file_str), add_write_permissions))
        if add_write_permissions:
            paths.add_write_permissions_for_path(str(current_dir / tar_file_str))
        if task_dir is not None:
            try:
                shutil.copyfile(str(current_dir / tar_file_str), str(task_dir / tar_file_str))
            except shutil.Error:
                pass
            resource = GenericTestResource(task, description, task_dir / tar_file_str)
        else:
            resource = GenericTestResource(task, description, current_dir / tar_file_str)
        resource.ttl = ttl
        if backup:
            resource.backup_task = True
        return resource

    def create_empty_resource(self, task, description, current_dir, resource_file_name, ttl=1, backup=False):
        tar_command = ["tar", "-cf", resource_file_name, "-T", "/dev/null"]
        self.shell_executor.execute_shell_and_check(
            args=tar_command,
            cwd=str(current_dir),
            timeout=1800
        )
        # experiment for https://st.yandex-team.ru/FRANKENSTEIN-997
        add_write_permissions = getattr(task.Parameters, 'add_write_permissions', True)
        logging.info("Need add_write_permissions for path {}? {}".format(
            str(current_dir / resource_file_name), add_write_permissions))
        if add_write_permissions:
            paths.add_write_permissions_for_path(str(current_dir / resource_file_name))
        resource = GenericTestResource(task, description, current_dir / resource_file_name)
        resource.ttl = ttl
        if backup:
            resource.backup_task = True
        return resource

    def add_to_resource(self, resource, path):
        TarResourceHelper.add_to_archive(str(sdk2.ResourceData(resource).path.as_posix()), str(path))

    @staticmethod
    def pack(archive, archive_folder, src):
        logger.info('Tar {} to {}/{}'.format(src, archive_folder, archive))
        dst_path = os.path.join(archive_folder, archive)
        try:
            with tarfile.open(dst_path, 'w') as arc_fh:
                if os.path.isfile(src):
                    arc_fh.add(src, arcname=os.path.basename(src))
                else:
                    for entry in os.listdir(src):
                        arc_fh.add(os.path.join(src, entry), arcname=entry)
        except Exception as e:
            logger.exception(e)
        logger.info("Tarred {} to {}".format(src, dst_path))

    @staticmethod
    def unpack(archive, folder):
        logger.info("Untar {} to {}".format(archive, folder))
        try:
            with tarfile.open(archive, 'r') as arc_fh:
                arc_fh.extractall(folder)
        except Exception as e:
            logger.exception(e)
        logger.info("Untarred {}".format(archive))
        logger.debug('Folder {} has unpacked content: {}'.format(folder, os.listdir(folder)))

    @staticmethod
    def add_to_archive(archive, src):
        logger.info('Add {} to archive {}'.format(src, archive))
        try:
            with tarfile.open(archive, 'a') as arc_fh:
                if os.path.isfile(src):
                    arc_fh.add(src, arcname=os.path.basename(src))
                else:
                    for entry in os.listdir(src):
                        arc_fh.add(os.path.join(src, entry), arcname=entry)
        except Exception as e:
            logger.exception(e)
        logger.info("Tarred {} to {}".format(src, archive))
