import distutils
import glob
import logging
import os

from sandbox import sdk2
from sandbox.common import fs
from sandbox.projects.common.teamcity import TeamcityArtifacts
from sandbox.projects.common.teamcity import TeamcityServiceMessagesLog
from sandbox.projects.metrika.mobile.sdk.helpers.ShellExecutor import ShellExecutor
from sandbox.sdk2 import paths


class TeamCityArtifactPublisher:

    def __init__(self, shell_executor=ShellExecutor()):
        self.shell_executor = shell_executor

    def create_teamcity_artifacts(self, task, base_dir, teamcity_artifact_paths, teamcity_messages_path, *args):
        teamcity_artifacts_path = self.make_teamcity_artifacts_dir(task, base_dir, teamcity_artifact_paths)
        self.publish_teamcity_artifacts(task, teamcity_messages_path, teamcity_artifacts_path)

    def publish_teamcity_artifacts(self, task, teamcity_messages_path, teamcity_artifacts_path, dest_path="."):
        with open(teamcity_messages_path, "a") as fout:
            fout.write("##teamcity[publishArtifacts '{} => {}']\n".format(teamcity_artifacts_path, dest_path))

        sdk2.ResourceData(TeamcityServiceMessagesLog(task, "TeamCity messages", teamcity_messages_path)).ready()
        sdk2.ResourceData(TeamcityArtifacts(task, "TeamCity artifacts", teamcity_artifacts_path)).ready()

    def make_teamcity_artifacts_dir(self, task, base_dir, teamcity_artifact_paths):
        teamcity_artifacts_dir = os.path.join(os.getcwd(), 'teamcity_artifacts')
        self._copy_logs(task, teamcity_artifacts_dir)
        old_cwd = os.getcwd()
        os.chdir(base_dir)

        for source, destination in teamcity_artifact_paths.iteritems():
            if destination.endswith(".zip"):
                destination = os.path.join(teamcity_artifacts_dir, destination.replace(".zip", ""))
            else:
                destination = os.path.join(teamcity_artifacts_dir, destination)
            if not os.path.exists(os.path.dirname(destination)):
                os.makedirs(os.path.dirname(destination))

            sources = glob.glob(source)
            logging.info("Found {} for {}".format(str(sources), source))

            if len(sources) > 1:
                if not os.path.exists(destination):
                    os.makedirs(destination)
                for artifact_source in sources:
                    real_destination = os.path.join(destination, os.path.basename(artifact_source))
                    if os.path.isfile(artifact_source):
                        logging.info("Copy file {} to {}".format(artifact_source, real_destination))
                        distutils.file_util.copy_file(artifact_source, real_destination)
                    elif os.path.isdir(artifact_source):
                        logging.info("Copy dir {} to {}".format(artifact_source, real_destination))
                        distutils.dir_util.copy_tree(artifact_source, real_destination)
            elif len(sources) == 1:
                if os.path.isfile(sources[0]):
                    if not os.path.exists(destination):
                        os.makedirs(destination)
                    real_destination = os.path.join(destination, os.path.basename(sources[0]))
                    logging.info("Copy file {} to {}".format(sources[0], real_destination))
                    distutils.file_util.copy_file(sources[0], real_destination)
                elif os.path.isdir(sources[0]):
                    logging.info("Copy dir {} as {}".format(sources[0], destination))
                    distutils.dir_util.copy_tree(sources[0], destination)

        for source, destination in teamcity_artifact_paths.iteritems():
            if destination.endswith(".zip"):
                destination = os.path.join(teamcity_artifacts_dir, destination)
                if os.path.exists(destination.replace(".zip", "")):
                    self._zip_and_remove(destination.replace(".zip", ""))
        os.chdir(old_cwd)
        return 'teamcity_artifacts'

    def _copy_logs(self, task, teamcity_artifacts_path):
        log_resources = sdk2.Resource.find(task=task, type='TASK_LOGS').first()
        logs_dir = teamcity_artifacts_path + '/logs'
        if not os.path.exists(teamcity_artifacts_path):
            os.makedirs(teamcity_artifacts_path)

        if log_resources:
            fs.copy_dir(sdk2.ResourceData(log_resources).path.as_posix(), logs_dir)
            # experiment for https://st.yandex-team.ru/FRANKENSTEIN-997
            add_write_permissions = getattr(task.Parameters, 'add_write_permissions', True)
            logging.info("Need add_write_permissions for path {}? {}".format(logs_dir, add_write_permissions))
            if add_write_permissions:
                paths.add_write_permissions_for_path(logs_dir)
            self._zip_and_remove(logs_dir)

    def _zip_and_remove(self, directory):
        zip_command = ["zip", "-r", "../" + os.path.basename(directory) + ".zip", "."]
        self.shell_executor.execute_shell_and_check(
            args=zip_command,
            cwd=directory,
            timeout=1800
        )
        distutils.dir_util.remove_tree(directory, verbose=True)
