# coding=utf-8

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.utils import Enum

BUILD_XCODE_PATH = '/Applications/Xcode11.3.app'
BUILD_DEVICE_OS = '11.3'
BUILD_DEVICE_MODEL = 'iPhone X'


class MacType(Enum):
    YMAKE_DEV_MAC = "ymake-dev-mac"
    MOBILE_MAC = "mobile-mac"


class IosCommonParameters(sdk2.Task.Parameters):
    """
    Общие параметры для выбора маков и их окружения для iOS прогонов
    """

    with sdk2.parameters.RadioGroup("Mac type") as mac_type:
        mac_type.values[MacType.YMAKE_DEV_MAC] = mac_type.Value(MacType.YMAKE_DEV_MAC, default=True)
        mac_type.values[MacType.MOBILE_MAC] = mac_type.Value(MacType.MOBILE_MAC)
    with mac_type.value[MacType.MOBILE_MAC]:
        use_catalina = sdk2.parameters.Bool("Use Mac OS Catalina [Deprecated. Use osx_tags]", default=True)
        use_big_sur = sdk2.parameters.Bool("Use Mac OS Big Sur [Deprecated. Use osx_tags]", default=False)
        osx_tags = sdk2.parameters.String("OSX_* tags. Sample: OSX_CATALINA | OSX_BIG_SUR. "
                                          "Available: https://nda.ya.ru/t/eoTx0fsA486m4o")
    jdk_version = sdk2.parameters.String("JDK version", default="8u202",
                                         description="Available versions: https://nda.ya.ru/t/4Rh1g6lN3nA8qp")
    rvm_plus_ruby_version = sdk2.parameters.String("Rvm+Ruby version",
                                                   description="Available versions: https://nda.ya.ru/t/G5j6Yfzw3nA9NL")

    @staticmethod
    def get(task):
        return {
            IosCommonParameters.mac_type.name: task.Parameters.mac_type,
            IosCommonParameters.use_catalina.name: task.Parameters.use_catalina,
            IosCommonParameters.use_big_sur.name: task.Parameters.use_big_sur,
            IosCommonParameters.osx_tags.name: task.Parameters.osx_tags,
            IosCommonParameters.jdk_version.name: task.Parameters.jdk_version,
            IosCommonParameters.rvm_plus_ruby_version.name: task.Parameters.rvm_plus_ruby_version,
        }


class IosRunnerParameters(sdk2.Task.Parameters):
    """
    Параметры окружения для iOS раннера
    """

    with sdk2.parameters.Group("Mac type parameters") as mac_type_group:
        ios_common_parameters = IosCommonParameters
    with sdk2.parameters.Group("iOS parameters") as ios_group:
        with ios_common_parameters.mac_type.value[MacType.MOBILE_MAC]:
            xcode_version = sdk2.parameters.String("Xcode version",
                                                   description="Available versions: "
                                                               "https://nda.ya.ru/t/9_Kdb0_y3mKvk7")
        with ios_common_parameters.mac_type.value[MacType.YMAKE_DEV_MAC]:
            xcode_path = sdk2.parameters.String("Xcode Path", required=True)
        device_os = sdk2.parameters.String("Device OS", required=False)
        device_model = sdk2.parameters.String("Device Model", required=False)

    @staticmethod
    def get(task):
        params = IosCommonParameters.get(task)
        params.update({
            IosRunnerParameters.xcode_version.name: task.Parameters.xcode_version,
            IosRunnerParameters.xcode_path.name: task.Parameters.xcode_path,
            IosRunnerParameters.device_os.name: task.Parameters.device_os,
            IosRunnerParameters.device_model.name: task.Parameters.device_model,
        })
        return params

    @staticmethod
    def get_from_launcher(task, simulator_config):
        params = IosCommonParameters.get(task)
        params.update({
            IosRunnerParameters.device_os.name: simulator_config['ios_version'],
            IosRunnerParameters.device_model.name: simulator_config['device_model'],
        })
        if simulator_config['mac_type'] == 'ymake':
            params.update({
                IosRunnerParameters.mac_type.name: MacType.YMAKE_DEV_MAC,
                IosRunnerParameters.xcode_path.name: simulator_config['xcode_path'],
            })
        elif simulator_config['mac_type'] == 'mobile':
            params.update({
                IosRunnerParameters.mac_type.name: MacType.MOBILE_MAC,
                IosCommonParameters.use_catalina.name: False,
                IosCommonParameters.use_big_sur.name: False,
                IosCommonParameters.osx_tags.name: simulator_config['osx_tags'],
                IosRunnerParameters.xcode_version.name: simulator_config['xcode_version'],
            })
        else:
            raise TaskError('Unknown mac type {}'.format(simulator_config['mac_type']))
        return params


class IosBuildParameters(IosRunnerParameters):
    """
    Параметры окружения для сборки iOS агента
    """

    @staticmethod
    def get_from_launcher(task):
        params = IosCommonParameters.get(task)
        params.update({
            IosBuildParameters.xcode_version.name: task.Parameters.build_xcode_version,
            IosBuildParameters.xcode_path.name: task.Parameters.build_xcode_path,
            IosBuildParameters.device_os.name: task.Parameters.build_device_os,
            IosBuildParameters.device_model.name: task.Parameters.build_device_model,
        })
        return params


class IosLauncherParameters(sdk2.Task.Parameters):
    """
    Параметры окружения для iOS лаунчера
    """

    with sdk2.parameters.Group("Build agent parameters") as build_agent_group:
        ios_common_parameters = IosCommonParameters
        with ios_common_parameters.mac_type.value[MacType.MOBILE_MAC]:
            build_xcode_version = sdk2.parameters.String("Xcode version to use for building agents", required=True,
                                                         description="Available versions: "
                                                                     "https://nda.ya.ru/t/9_Kdb0_y3mKvk7")
        with ios_common_parameters.mac_type.value[MacType.YMAKE_DEV_MAC]:
            build_xcode_path = sdk2.parameters.String("XCode path to use for building agents",
                                                      default=BUILD_XCODE_PATH)
        build_device_os = sdk2.parameters.String("Device OS to use for building agents",
                                                 default=BUILD_DEVICE_OS)
        build_device_model = sdk2.parameters.String("Device model to use for building agents",
                                                    default=BUILD_DEVICE_MODEL)
