import datetime
import logging
import os
import six

import sandbox.sdk2.helpers.process
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types.misc import DnsType
from sandbox.projects.resource_types import TASK_CUSTOM_LOGS
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.metrika.utils import custom_report_logger


class RunTestsBase(sdk2.Task):
    """
    Sdk Tests Base
    """

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64
        disk_space = 16384

    class Parameters(sdk2.Task.Parameters):
        expires = datetime.timedelta(hours=24)
        report_duration = sdk2.parameters.Integer("Report stage duration in seconds", required=True,
                                                  default_value=600)

    class Context(sdk2.Task.Context):
        artifacts_resource_id = None

    @property
    def console_logger(self):
        try:
            return self._shell_logger
        except AttributeError:
            self._shell_logger = logging.getLogger("console")

        return self._shell_logger

    @property
    def logger(self):
        try:
            return self._logger
        except AttributeError:
            self._logger = logging.getLogger("scenario")
        return self._logger

    @property
    def env(self):
        try:
            return self._env
        except AttributeError:
            self._env = dict(l.split('=', 1) for l in
                             subprocess.check_output(['/bin/bash', '-c', '. /etc/profile && printenv']).splitlines())
            self._env['LANG'] = 'en_US.UTF-8'
        return self._env

    @property
    def execute_shell_timeout(self):
        exec_time = self.server.task.current.read()["execution"]["current"]
        remaining_time = self.Parameters.kill_timeout - exec_time
        timeout = remaining_time - self.Parameters.report_duration
        self.logger.info("exec_time %d sec, remains %d sec, shell timeout: %d sec", exec_time, remaining_time, timeout)
        return timeout

    @property
    def clean_up_timeout(self):
        exec_time = self.server.task.current.read()["execution"]["current"]
        timeout = self.Parameters.kill_timeout - exec_time
        self.logger.info("exec_time %d sec, clean up timeout: %d sec", exec_time, timeout)
        return timeout

    def log_path_custom(self, *path):
        """
        Build path relative to custom logs - see log_path
        :param path: path components
        :rtype: pathlib2.Path
        """
        p = self.path("artifacts", *path)
        if not p.exists():
            os.makedirs(str(p))
        return p

    def _discard_expire_time(self):
        self.Parameters.expires = None

    def _execute_shell(self, args, cwd=None, env=None, timeout=None):
        """
        :param args: command-line arguments, single string or sequence
        :param cwd: current dir
        :param env: environment variables
        :return: exit code
        """
        with sandbox.sdk2.helpers.ProcessLog(self,
                                             logger=self.console_logger,
                                             set_action=False,
                                             stdout_level=logging.DEBUG,
                                             stderr_level=logging.ERROR) as pl:
            pl.logger.info("\t\tCMD: %s", args)
            process = sandbox.sdk2.helpers.subprocess.Popen(args, cwd=cwd, env=env,
                                                            stdout=pl.stdout,
                                                            stderr=pl.stderr)
            exit_code = process.wait(timeout=timeout)

            pl.logger.info("\t\tEXIT CODE: %s", exit_code)

            return exit_code

    def _execute_shell_and_check(self, args, cwd=None, env=None, timeout=None):
        """
        if exit code is not zero throws exception
        :param args: command-line arguments, single string or sequence
        :param cwd: current dir
        :param env: environment variables
        :return: nothing
        """
        exitcode = self._execute_shell(args, cwd=cwd, env=env, timeout=timeout)

        if exitcode:
            raise TaskFailure(
                "Command\n{!r}\nFAILED. Exit code: {!r}".format(args, exitcode))

    def _execute_gradle(self, path, tasks, gradle_progerties=None, system_properties=None, timeout=None):
        """
        Launch gradlew
        :param path: path to build.gradle - current dir
        :param system_properties: dictionary of system properties -Dk=v...
        :param gradle_progerties: dictionary of gradle properties -Pk=v....
        :param tasks: tasks and other parameters
        :return:
        """
        args = ["./gradlew"]
        if system_properties:
            args += ["-D{0}={1}".format(k, v) for (k, v) in system_properties.iteritems()]
        if gradle_progerties:
            args += ["-P{0}={1}".format(k, v) for (k, v) in gradle_progerties.iteritems()]
        if isinstance(tasks, six.string_types):
            args += [tasks]
        else:
            args += list(tasks)
        return self._execute_shell_and_check(args, path, env=self.env, timeout=timeout or self.execute_shell_timeout)

    def _create_custom_logs_resource(self):
        self.Context.artifacts_resource_id = TASK_CUSTOM_LOGS(self, "Artifacts", str(self.log_path_custom())).id

    @sdk2.header()
    @custom_report_logger
    def header(self):
        if self.Context.artifacts_resource_id is not None:
            proxy = sdk2.Resource[self.Context.artifacts_resource_id].http_proxy
            footer = []
            append_artifacts_footer(footer, proxy)
            return footer
        else:
            return None

    def _get_reports_url(self):
        return "https://proxy.sandbox.yandex-team.ru/task/{0}/artifacts/reports".format(self.id)


def append_artifacts_footer(footer, resource_link):
    if resource_link:
        footer.append({
            "content": {
                "<h4>Artifacts</h4>": {
                    "header": [
                        {"key": "html", "title": ""},
                        {"key": "allure", "title": ""},
                        {"key": "jacoco", "title": ""},
                    ],
                    "body": {
                        "html": ["<a href='{proxy!s}/reports/html/index.html' target='_blank'>Gradle Report</a>".format(proxy=resource_link)],
                        "allure": ["<a href='{proxy!s}/reports/allure/index.html' target='_blank'>Allure Report</a>".format(proxy=resource_link)],
                        "jacoco": [
                            "<a href='{proxy!s}/reports/jacoco/reportTestCoverage/html/index.html' target='_blank'>Jacoco report</a>".format(
                                proxy=resource_link)],
                    }
                }
            },
            "helperName": ""
        })
