import os

import pathlib2
import six

from sandbox import sdk2
from sandbox.common.utils import Enum
from sandbox.sandboxsdk.environments import TarballToolkitBase


class PlatformKindIos(Enum):
    Enum.preserve_order()

    NATIVE = None
    UNITY = None
    XAMARIN = None


class PlatformKindAndroid(Enum):
    Enum.preserve_order()

    UNITY = None
    XAMARIN = None


class Rvm(sdk2.Resource):
    """
        Tarball with installed RVM
    """
    any_arch = False
    version = sdk2.Attributes.String("version", required=True)
    platform = sdk2.Attributes.String("platform", required=True)


class RvmEnvironment(TarballToolkitBase):
    """
        RVM: Ruby enVironment Manager
        https://rvm.io/
    """
    resource_type = Rvm
    name = "rvm"
    sys_path_utils = ['rvm', ]

    STABLE = "1.29.8"

    DOT_FILE = ".bash_profile"

    def __init__(self, version=None, **kws):
        super(RvmEnvironment, self).__init__(version or RvmEnvironment.STABLE, **kws)

    def fix_dot_file(self, rvm_dir):
        dot_file = pathlib2.Path.home().joinpath(RvmEnvironment.DOT_FILE)
        self.logger.info("Dot file: {}", dot_file)
        if dot_file.exists():
            self.logger.warning("Dot file {} exists. Will be overwritten. Content:\n{}", dot_file, dot_file.read_text())

        dot_file_content = "source " + pathlib2.Path(rvm_dir).joinpath("scripts/rvm").as_posix()
        self.logger.info("Dot file {} content:\n{}", dot_file, dot_file_content)
        dot_file.write_text(six.text_type(dot_file_content))

    def prepare(self):
        env_dir = super(RvmEnvironment, self).prepare()
        bin_dir = os.path.join(env_dir, 'bin')
        self.update_os_path_env(bin_dir)
        self.update_os_env('rvm_bin_path', bin_dir)
        self.update_os_env('rvm_path', env_dir)
        self.update_os_env('rvm_prefix', os.path.dirname(env_dir))
        self.update_os_env('rvm_version', self.version)
        self.update_os_env('SSL_CERT_FILE', os.path.join(env_dir, 'cacert.pem'))
        self.check_environment()
        self.fix_dot_file(env_dir)
        return env_dir
