# coding=utf-8
import logging

import pathlib2

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.projects.metrika.mobile.sdk.utils import RvmEnvironment
from sandbox.projects.metrika.mobile.sdk.helpers.DistributionCertificateHelper import DistributionCertificateHelper
from sandbox.projects.metrika.mobile.sdk.helpers.GitHelper import GitHelper
from sandbox.projects.metrika.mobile.sdk.helpers.ProvisioningHelper import ProvisioningHelper
from sandbox.projects.metrika.mobile.sdk.helpers.ShellExecutor import ShellExecutor
from sandbox.projects.metrika.mobile.sdk.helpers.TeamCityArtifactPublisher import TeamCityArtifactPublisher
from sandbox.projects.metrika.mobile.sdk.helpers.VcsHelper import VcsHelper
from sandbox.projects.metrika.mobile.sdk.parameters.TeamcityParameters import TeamcityParameters
from sandbox.projects.metrika.mobile.sdk.parameters.VcsParameters import VcsParameters
from sandbox.projects.common.decorators import retries

XAMARIN_SETTINGS_PATH = str(pathlib2.Path.home().joinpath('Library/Preferences/Xamarin/Settings.plist'))

CERTIFICATE_PATH = 'certificate.pem'


class XamarinMsbuildRunner(sdk2.task.Task):
    """
    Run MSBuild for Xamarin project
    """

    class Utils:
        vcs_helper = VcsHelper()
        git_helper = GitHelper()
        shell_executor = ShellExecutor()
        provisioning_helper = ProvisioningHelper(git_helper)
        distribution_certificate_helper = DistributionCertificateHelper(shell_executor)
        teamcity_artifact_publisher = TeamCityArtifactPublisher(shell_executor)

    class Requirements(sdk2.Task.Requirements):
        dns = DnsType.DNS64
        disk_space = 16384
        client_tags = Tag.CUSTOM_OSX_JAVA & ~Tag.USER_BROKE
        environments = [RvmEnvironment()]

    class Parameters(sdk2.Task.Parameters):
        vcs = VcsParameters
        with sdk2.parameters.Group("Параметры MSBuild") as msbuild:
            project_path = sdk2.parameters.String("Путь к проекту",
                                                  description="Путь до рабочей области проекта относительно корня репозитория",
                                                  default="")
            project_file = sdk2.parameters.String("Файл проекта или каталог",
                                                  description="собирает указанные целевые объекты в файле проекта. "
                                                              "Если файл проекта не указан, MSBuild ищет в текущем рабочем каталоге файл, который имеет расширение, "
                                                              "заканчивающееся на proj, и использует этот файл. Если указан каталог, MSBuild ищет файл проекта в нем")
            target = sdk2.parameters.String("Target параметры",
                                            description="Перечиляются через запятую или точку с запятой. Например: 'Resources;Compile'",
                                            required=True)
            properties = sdk2.parameters.Dict("Свойства проекта",
                                              description="Задать или переопределить конкретные свойства уровня проекта")
            other_parameters = sdk2.parameters.List("Параметры сборки",
                                                    description="Остальные параметры сборки. Например, 'noconsolelogger' или в краткой форме 'noconlog'")

        with sdk2.parameters.Group("Параметры для iOS") as ios:
            xcode_path = sdk2.parameters.String("Путь к Xcode",
                                                description="Устанавливается в настроки Visual Studio")
            import_provisioning = sdk2.parameters.Bool("Импортировать provisioning файлы",
                                                       description="Если установлен в True, то скачиваются provisioning файлы",
                                                       default=False)
            certificate = sdk2.parameters.Vault("Сертификат для подписи приложения",
                                                description="Сертификат, который берётся из vault, значение необходимо задавать в формате owner:name")

        with sdk2.parameters.Group("Другое") as other:
            teamcity_build_id = sdk2.parameters.String("TeamCity build id",
                                                       description="Не используется. Устанавливается TeamCity")
        teamcity_parameters = TeamcityParameters

    class Context(sdk2.Task.Context):
        pass

    @retries(5)
    def on_prepare(self):
        with sdk2.helpers.ProgressMeter("Checkout"):
            self.repo = self.Utils.vcs_helper.clone_with_task(self)
        with sdk2.helpers.ProgressMeter("Clone provisioning"):
            self.Utils.provisioning_helper.clone_provisioning(self._work_dir(), self.Parameters.ssh_key)

    def on_execute(self):
        with self.repo:
            try:
                with sdk2.helpers.ProgressMeter("Import provisioning files"):
                    self.Utils.provisioning_helper.prepare_provisioning(self.logger, self._work_dir())
                with sdk2.helpers.ProgressMeter("Import certificate"):
                    password = sdk2.Vault.data('METRIKA', 'YMAKE_DEV_MAC_SANDBOX_USER_PASSWORD')
                    self.Utils.distribution_certificate_helper.import_certificate(
                        certificate=self.Parameters.certificate,
                        password=password,
                        work_dir=self._project_dir()
                    )
                with sdk2.helpers.ProgressMeter("Restore nuget packages"):
                    self._restore_nuget_package()
                with sdk2.helpers.ProgressMeter("Run MSBuild"):
                    self._msbuild()
            except Exception:
                logging.error("Exception in scenario", exc_info=True)
                raise
            finally:
                with sdk2.helpers.ProgressMeter("Archive artifacts"):
                    self._archive_artifacts()
                with sdk2.helpers.ProgressMeter("Remove files"):
                    self.Utils.provisioning_helper.remove_files()

    def _work_dir(self, *path):
        return str(self.path("wd", *path))

    def _project_dir(self, *path):
        return str(self._work_dir(self.Parameters.project_path, *path))

    def _restore_nuget_package(self):
        self.Utils.shell_executor.execute_shell_and_check(
            args=['nuget', 'restore'],
            cwd=self._project_dir(),
        )

    def _msbuild(self):
        args = ['msbuild']
        args += ["/property:{0}={1}".format(k, self._fix_parameter(v)) for (k, v) in self.Parameters.properties.iteritems()] if self.Parameters.properties else []
        args += ["/{0}".format(v) for v in self.Parameters.other_parameters] if self.Parameters.other_parameters else []
        args += ["/target:{0}".format(self.Parameters.target)]
        args += [self.Parameters.project_file] if self.Parameters.project_file else []

        env = self.Utils.shell_executor.default_env
        if self.Parameters.xcode_path:
            env['MD_APPLE_SDK_ROOT'] = self.Parameters.xcode_path

        self.Utils.shell_executor.execute_shell_and_check(args=args, cwd=self._project_dir(), env=env)

    def _fix_parameter(self, parameter):
        if parameter.startswith('$'):
            return getattr(self.Parameters, parameter[1:])
        else:
            return parameter

    def _archive_artifacts(self):
        if self.Parameters.teamcity_artifact_paths:
            self.Utils.teamcity_artifact_publisher.create_teamcity_artifacts(self, self._project_dir(),
                                                                             self.Parameters.teamcity_artifact_paths,
                                                                             str(self.path("teamcity_messages.log")))

    @property
    def logger(self):
        try:
            return self._logger
        except AttributeError:
            self._logger = logging.getLogger("scenario")
        return self._logger
