import os

import sandbox.common.types.misc as ctm
import sandbox.common.types.task as ctt
from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.utils import get_resource_link, get_task_link
from sandbox.projects.sandbox import sandbox_lxc_image

from sandbox.projects.metrika.utils import custom_report_logger


def load_file(filename):
    filename = os.path.join(os.path.dirname(os.path.realpath(__file__)), filename)
    with open(filename) as fh:
        return fh.read()


class CreateUnityEnvironmentPreparerContainer(sdk2.Task):
    """
        Create container for unity environment preparer
    """

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        cores = 1
        ram = 1 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Task.Parameters):
        ruby_version = sdk2.parameters.String("Ruby version", default="2.7.0")
        u3d_version = sdk2.parameters.String("u3d version", default="1.2.1")

    class Context(sdk2.Task.Context):
        create_image_task_id = None
        container_id = None

    def _get_task(self):
        script = load_file('prepare_container_template.sh').format(
            self.Parameters.ruby_version, self.Parameters.u3d_version
        )
        description = 'Create container for unity environment preparer'
        image_attrs = {
            'target': 'unity_environment_preparer',
        }
        params = {
            sandbox_lxc_image.SandboxLxcImage.Parameters.resource_description.name: 'Unity environment preparer container',
            sandbox_lxc_image.SandboxLxcImage.Parameters.ubuntu_release.name: sandbox_lxc_image.UbuntuRelease.XENIAL,
            sandbox_lxc_image.SandboxLxcImage.Parameters.test_result_lxc.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_attrs.name: image_attrs,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_image.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.install_common.name: True,
            sandbox_lxc_image.SandboxLxcImage.Parameters.custom_script.name: script,
        }
        return sandbox_lxc_image.SandboxLxcImage(
            self,
            description=description,
            priority=min(
                self.Parameters.priority,
                ctt.Priority(ctt.Priority.Class.SERVICE, ctt.Priority.Subclass.NORMAL)  # default API limitation
            ),
            **params
        )

    def on_execute(self):
        with self.memoize_stage.create_image(commit_on_entrance=False):
            self.Context.create_image_task_id = self._get_task().enqueue().id
            raise sdk2.WaitTask(
                self.Context.create_image_task_id,
                tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK)
            )
        with self.memoize_stage.check_result(commit_on_entrance=False):
            if sdk2.Task[self.Context.create_image_task_id].status not in ctt.Status.Group.SUCCEED:
                raise TaskFailure('Not LXC containers built (see subtasks for details)')
        with self.memoize_stage.get_container_info(commit_on_entrance=False):
            self.Context.container_id = sdk2.Task[self.Context.create_image_task_id].Context.container_id

    @sdk2.header()
    @custom_report_logger
    def header(self):
        header = ''
        if self.Context.create_image_task_id:
            header += 'Task of creating container: <a href="{}">{}</a>\n'.format(
                get_task_link(self.Context.create_image_task_id), self.Context.create_image_task_id
            )
        if self.Context.container_id:
            header += '<br>Container: <a href="{}">{}</a>\n'.format(
                get_resource_link(self.Context.container_id), self.Context.container_id
            )
        return '<span style="font-size: 15px;">{}</span>'.format(header)
