import logging
import pkgutil

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.common.types.resource import State
from sandbox.common.utils import get_resource_link
from sandbox.projects.common import binary_task
from sandbox.sdk2.service_resources import LxcContainer

from sandbox.projects.metrika.mobile.unity.unity_environment import Unity
from sandbox.projects.metrika.utils import custom_report_logger

_logger = logging.getLogger("unity_environment_preparer")

UNITY_DIR = "unity"
ARCH_MAP = {
    "linux": ctm.OSFamily.LINUX,
    "mac": ctm.OSFamily.OSX,
}
PLATFORM_MAP = {
    "linux": "linux",
    "mac": "darwin",
}


class UnityEnvironmentPreparer(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
        Creates resource with Unity environment of given version
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.GENERIC
        dns = ctm.DnsType.DNS64
        cores = 4
        ram = 12 * 1024

        class Caches(sdk2.Requirements.Caches):
            pass  # Do not use any shared caches (required for running on multislot agent)

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.RadioGroup("Platform") as platform:
            platform.values["linux"] = platform.Value("Linux", default=True)
            platform.values["mac"] = platform.Value("MacOS")
        version = sdk2.parameters.String("Unity version",
                                         description="Available versions: `u3d available` or "
                                                     "https://dragonbox.github.io/unities/v1/versions.json",
                                         required=True)
        with sdk2.parameters.CheckGroup("Packages") as packages:
            packages.values["Unity"] = packages.Value("Unity", checked=True)
            packages.values["Android"] = packages.Value("Android")
            packages.values["iOS"] = packages.Value("iOS")

        binary_release = binary_task.binary_release_parameters(stable=True)

    class Context(sdk2.Task.Context):
        unity_resource_id = None

    def on_save(self):
        super(UnityEnvironmentPreparer, self).on_save()
        self.Requirements.container_resource = LxcContainer.find(
            state=State.READY,
            attrs={"target": "unity_environment_preparer"}
        ).first()

    def on_execute(self):
        super(UnityEnvironmentPreparer, self).on_execute()
        with sdk2.helpers.ProgressMeter("Prepare Unity"):
            self._prepare_unity(self.path(UNITY_DIR))
        with sdk2.helpers.ProgressMeter("Create Resource"):
            self._create_resource(self.path(UNITY_DIR))

    def _prepare_unity(self, dest):
        packages_str = ",".join(self.Parameters.packages)
        _logger.info("Prepare unity")
        _logger.info("Unity version: {}".format(self.Parameters.version))
        _logger.info("Platform: {}".format(self.Parameters.platform))
        _logger.info("Packages: {}".format(packages_str))

        script_file = "script.sh"
        with open(script_file, 'w') as f:
            f.write(pkgutil.get_data(__package__, "script.sh"))
        sdk2.helpers.process.subprocess.check_call([
            "/bin/bash", script_file, self.Parameters.version, self.Parameters.platform, packages_str, str(dest)
        ], env=self._get_env())

    def _create_resource(self, unity_dir):
        packages_str = ",".join(self.Parameters.packages)
        unity = Unity(
            self, "Unity v{} with {} for {}".format(self.Parameters.version, packages_str, self.Parameters.platform),
            unity_dir,
            arch=ARCH_MAP[self.Parameters.platform],
            version=self.Parameters.version,
            platform=PLATFORM_MAP[self.Parameters.platform],
            packages=packages_str,
        )
        for package in self.Parameters.packages:
            unity.__setattr__("package_{}".format(package), True)
        self.Context.unity_resource_id = unity.id
        sdk2.ResourceData(unity).ready()

    @sdk2.header()
    @custom_report_logger
    def header(self):
        header = ''
        if self.Context.unity_resource_id:
            header += "Unity: <a href=\"{}\">{}</a>\n".format(
                get_resource_link(self.Context.unity_resource_id), self.Context.unity_resource_id
            )
        return "<span style=\"font-size: 15px;\">{}</span>".format(header)

    @staticmethod
    def _get_env():
        return dict(_.split("=", 1) for _ in sdk2.helpers.subprocess.check_output(
            ["/bin/bash", "-c", ". /etc/profile && printenv"]).splitlines())
