# coding=utf-8
import cgi
import inspect
import logging
import traceback
from functools import wraps
from os import path

import six

import sandbox.common.types.task as ctt
from sandbox import common, sdk2
from sandbox.common import fs
from sandbox.projects.metrika.utils import settings
from sandbox.sdk2 import yav

TASK_FINISHED_STATUSES = ctt.Status.Group.FINISH | ctt.Status.Group.BREAK


def bump_version(old_version, bumped_part_number=None):
    version_parts = map(int, old_version.split('.'))
    if bumped_part_number == 'last':
        bumped_part_number = None

    if bumped_part_number is not None and bumped_part_number >= len(version_parts):
        raise IndexError("Current version %s has only %d parts, bumped part doesn't exist" % (old_version, len(version_parts)))

    if bumped_part_number is None or bumped_part_number == len(version_parts) - 1:
        version_parts[-1] += 1
    else:
        version_parts[bumped_part_number] += 1
        for i in range(bumped_part_number + 1, len(version_parts)):
            version_parts[i] = 0

    return '.'.join(map(str, version_parts))


class MetrikaLastReleasedResource(sdk2.parameters.Resource):
    @common.utils.classproperty
    def default_value(cls):
        resources = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            owner='METRIKA',
            attrs={'released': 'stable'},
            limit=1
        )['items']
        if not resources:
            return None
        return resources[0]['id']


def print_ticket(task, key, comment='Тикет'):
    task.set_info('{0}: <a href="https://st.yandex-team.ru/{1}">{1}</a>'.format(comment, key), do_escape=False)


class CommonParameters(sdk2.Parameters):
    owner = settings.owner


def read_sources_file(name, dir=None):
    if not dir:
        stack = inspect.stack()
        i = 0
        # .pyc != .py
        while path.splitext(path.abspath(stack[i][1]))[0] == path.splitext(path.abspath(__file__))[0] or 'projects/metrika' not in stack[i][1]:
            i += 1
        dir = path.dirname(stack[i][1])
    full_path = path.join(dir, name)
    logging.debug("Read file from: %s", full_path)
    return fs.read_file(full_path)


def get_template(template, dir=None):
    import jinja2
    return jinja2.Environment(loader=jinja2.BaseLoader(), extensions=['jinja2.ext.do']).from_string(
        read_sources_file(template, dir)
    )


def render(template, context={}, dir=None):
    return get_template(template, dir).render(context)


def get_duty(project, group):
    from metrika.pylib import duty
    duty_api = duty.DutyAPI(token=yav.Secret(settings.yav_uuid).data()["oauth-token"])
    return duty_api.get_duty_group(project=project, duty_group_key=group).duty


def decode(string):
    """
    декодирует строку или иной объект в unicode-строку для отображения в интерфейсе
    :param string:
    :return: unicode-строка
    """
    return six.ensure_text(string, encoding="utf-8", errors="replace")


def encode(string):
    """
    Кодирует строку или иной объект в 8-битную строку для передачи куда надо
    :param string:
    :return: 8-битная строка
    """
    return six.ensure_str(string, encoding="utf-8", errors="replace")


def display(value, default=""):
    """
    :param value:
    :param default:
    :return:
    """
    return decode(value or default)


def shell(task, args):
    with sdk2.helpers.ProcessLog(task, logger="console", stdout_level=logging.INFO, stderr_level=logging.ERROR) as pl:
        sdk2.helpers.subprocess.check_call(args, stdout=pl.stdout, stderr=sdk2.helpers.subprocess.STDOUT)


def custom_report_logger(f):
    @wraps(f)
    def wrapper(*args, **kwargs):
        try:
            return f(*args, **kwargs)
        except:
            message = traceback.format_exc()
            logging.debug(message)
            return '<pre>' + cgi.escape(message) + '</pre>'

    return wrapper
