# coding=utf-8
import requests

from sandbox.projects.release_machine.helpers import arcanum_helper

from sandbox.projects.common.decorators import retries


class ArcanumApi(arcanum_helper.ArcanumApi):
    @retries(5)
    def _do_get(self, url, params=None):
        return super(ArcanumApi, self)._do_get(url, params=params)

    @retries(5)
    def _do_post(self, url, data=None, files=None):
        return super(ArcanumApi, self)._do_post(url, data=data, files=files)

    @retries(5)
    def _do_put(self, url, data=None, files=None):
        return super(ArcanumApi, self)._do_put(url, data=data, files=files)

    def get_tree(self, path):
        return self._do_get_json("tree/node/trunk/arcadia/{}".format(path))

    def get_nodes(self, path):
        return [children.get("name") for children in self.get_nodes_raw(path)]

    def get_nodes_raw(self, path):
        return self.get_tree(path).get("children")

    def get_blob(self, path):
        return self._do_get_raw("v2/repos/arc/downloads?at=trunk&path={}".format(path))

    def get_history(self, path):
        return self._do_get_json("tree/history/trunk/arcadia/{}".format(path))

    def get_branches(self, name=""):
        return [branch.get("name") for branch in self._do_get_json("v2/repos/arc/branches?fields=name&name={}".format(name)).get("data")]

    def branch_exists(self, name):
        return name in self.get_branches(name)

    def _do_get_json(self, url, params=None):
        return self._do_get(url, params=params).json()

    def _do_get_raw(self, url, params=None):
        return self._do_get(url, params=params).text

    def _check_response(self, response, request_type):
        self.logger.debug("Checking response of url: %s", response.url)
        self.logger.debug("Response headers: %s", response.headers)
        self.logger.debug("Return code: %s. Content:\n%s", response.status_code, response.content)
        if response.status_code in {requests.codes.ok, requests.codes.created, requests.codes.no_content}:
            return response
        else:
            self.logger.error("{} request fails with status code {}: {}".format(
                request_type, response.status_code, response.content
            ))
            raise Exception('Error in arcanum: [{}] {}'.format(response.status_code, response.text))

    def pr_create(self, data, fields=[]):
        """
        Метод публичного API Arcanum - https://docs.yandex-team.ru/arcanum/communication/public-api
        https://a.yandex-team.ru/api/swagger/index.html#/public_api/createPullRequest
        :param data:
        :param fields:
        :return:
        """
        url = "v1/pull-requests"
        if fields:
            url += "?fields=" + ",".join(fields)
        return self._do_post(url, data=data).json()

    def pr_comment(self, id, content, issue=False):
        """
        Метод публичного API Arcanum - https://docs.yandex-team.ru/arcanum/communication/public-api
        https://a.yandex-team.ru/api/swagger/index.html#/public_api/getPullRequest
        :param id:
        :param content:
        :param issue:
        :return:
        """
        url = "v1/pull-requests/{}/comments".format(id)
        return self._do_post(url, data={"content": content, "issue": issue})

    def pr_get(self, id, fields=[]):
        """
        Метод публичного API Arcanum - https://docs.yandex-team.ru/arcanum/communication/public-api
        https://a.yandex-team.ru/api/swagger/index.html#/public_api/createComment
        :param id:
        :param fields:
        :return:
        """
        url = "v1/pull-requests/{}".format(id)
        return self._do_get_json(url, params={"fields": ",".join(fields)})
