# coding=utf-8
import logging
import os
import re
import shutil

import six
from sandbox.projects.resource_types import TASK_CUSTOM_LOGS

from sandbox import sdk2


def list_artifacts(root_dir, *patterns):
    """
    Сканирует указанный каталог на предмет поиска в нём артефактов
    :param root_dir: каталог для поиска
    :param patterns: регулярные выражения на относительный от корня путь к файлу
    :return: список относительных относительно root_dir путей файлов которые попали под regexps
    """
    result = []
    logging.info('list artifacts from %s\npatterns: %s', root_dir, patterns)
    if patterns:
        for root, dirs, files in os.walk(root_dir):
            for name in files:
                full_name = os.path.relpath(os.path.join(root, name), root_dir)
                for pattern in patterns:
                    if re.search(pattern, full_name):
                        logging.info(full_name)
                        result.append(full_name)
                        break
    return result


def archive_artifacts(task, target_dir, artifacts_dir, resource_cls, *artifacts, **attrs):
    """
    Перекладывает из target_dir в artifacts_dir то, что перечислено в artifacts и регистрирует artifacts_dir
    как ресурс TASK_CUSTOM_LOGS. Проглатывает и логирует все исключения, т.к. обычно вызывается в конце из finally
    и падения здесь не должны скрывать исключения в сценарии задачи.
    :param Task task: объект задачи
    :param Path | str target_dir: каталог, откуда собрать артефакты
    :param Path | str artifacts_dir: целевой каталог
    :param Resource resource_cls: класс ресурса
    :param artifacts: файлы и каталоги относительно target_dir, тут могут быть как строки, так и списки строк
    :return: id созданного ресурса
    """

    target_dir = str(target_dir)
    artifacts_dir = str(artifacts_dir)
    logging.info("archive artifacts to %s from %s", artifacts_dir, target_dir)
    if not os.path.exists(artifacts_dir):
        os.makedirs(artifacts_dir)

    if os.path.exists(target_dir):
        if artifacts:
            for artifact in artifacts:
                if isinstance(artifact, six.string_types):
                    artifact = [artifact]
                for single_artifact in artifact:
                    try:
                        dst_path = os.path.join(artifacts_dir, single_artifact)
                        dst_dir = os.path.dirname(dst_path)
                        if not os.path.exists(dst_dir):
                            os.makedirs(dst_dir)
                        src_path = os.path.join(target_dir, single_artifact)
                        copy = shutil.copytree if os.path.isdir(src_path) else shutil.copy
                        copy(src_path, dst_path)
                    except:
                        logging.error(
                            "Exception in archive artifact. target_dir=%(target_dir)s; artifact=%(artifact)s; artifacts_dir=%(artifacts_dir)s. Ignore",
                            {'target_dir': target_dir, 'artifact': single_artifact, 'artifacts_dir': artifacts_dir},
                            exc_info=True
                        )
        else:
            try:
                shutil.copytree(target_dir, artifacts_dir)
            except:
                logging.error(
                    "Exception in archive artifacts. target_dir=%(target_dir)s; artifacts_dir=%(artifacts_dir)s. Ignore",
                    {'target_dir': target_dir, 'artifacts_dir': artifacts_dir}, exc_info=True
                )
    return archive_artifacts_inplace(task, artifacts_dir, resource_cls, **attrs)


def archive_artifacts_inplace(task, artifacts_dir, resource_cls=TASK_CUSTOM_LOGS, **attrs):
    logging.info("archive artifacts in-place from %s", artifacts_dir)
    description = attrs.pop('description', os.path.basename(artifacts_dir))
    resource_cls.__default_attribute__ = sdk2.Attributes.String
    return resource_cls(task, description, artifacts_dir, **attrs).id
