# coding=utf-8
from __future__ import unicode_literals

import inspect
import logging
import sys
import traceback
from functools import partial

import six

from sandbox import sdk2
from sandbox.common import config
from sandbox.common.types import misc
from sandbox.common.urls import get_task_link
from sandbox.projects.common import binary_task
from sandbox.projects.common.binary_task import LastRefreshableBinary
from sandbox.projects.common.decorators import memoize
from sandbox.projects.metrika.utils.mixins.autoticket import AutoTicketMixin
from sandbox.projects.metrika.utils.mixins.subtasks import SubtasksMixin
from sandbox.projects.metrika.utils.requirements import MetrikaTinyRequirements, MetrikaLargeRequirements
from sandbox.sdk2 import Task

WRAPPED_FUNCTIONS = [
    var.__name__ for var in Task.__dict__.values()
    if inspect.isfunction(var) and var.__name__.startswith('on_')
]

TELEGRAM_IMG_BASE64 = """\
AAABAAEAICAAAAEAIACoEAAAFgAAACgAAAAgAAAAQAAAAAEAIAAAAAAAABAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA1aorBtSeK0fTmieV05wov9Kb
KdnSmijz05sp89GbKNnSmyi/05onldSaKEfVqisGAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA1Z8rGNOcKZbTnCjx0pso/9Kb
KP/Smyj/0pso/9KbKP/Smyj/0pso/9KbKP/Smyj/0pso/9OcKPHTnCmW1Z8rGAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA27ZJB9SdKYLTnSn005wp/9Oc
Kf/Tmyj/05wp/9ObKP/Tmyj/05so/9ObKP/Tmyj/05so/9ObKP/Tmyj/05so/9ObKP/TnCn01J0pgtu2
SQcAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAANWfKxjVnCnH1Jwp/9Sc
Kf/UnCn/1Jwp/9SdKf/UnCn/1J0p/9SdKf/UnCn/1J0p/9ScKf/UnCn/1J0p/9ScKf/UnSn/1J0p/9Sc
Kf/UnSn/1Zwpx9WfKxgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAADYoSwu1Z0r49Wd
Kv/UnCn/1Z0q/9ScKf/UnCn/1Z0q/9ScKf/VnSr/1Z0q/9ScKf/VnSr/1Jwp/9ScKf/UnSn/1Jwp/9Sd
Kf/UnSn/1Z0p/9SdKf/VnSn/1Z0q49efLS0AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA358rGNee
K+PVnSr/1Z0q/9aeKv/VnSr/1p4q/9aeKv/VnSr/1p4q/9WdKv/VnSr/1Z0q/9WdKv/VnSr/1Z0q/9ae
Kv/VnSr/1p4q/9aeKv/VnSr/1p4q/9WdKv/VnSr/154r49+fKxgAAAAAAAAAAAAAAAAAAAAAAAAAANu2
SQfXnyrH1p4r/9aeKv/Wnir/1p4q/9aeKv/Wnir/1p4q/9eeK//Wnir/154r/9eeK//Xnyv/154r/9ef
K//Xnyv/1p4q/9efK//Wnir/1p4q/9aeK//Wnir/1p4r/9aeK//Wnir/158sx9u2SQcAAAAAAAAAAAAA
AAAAAAAA2KErgtifK//Xnyv/2J8r/9ifK//Xnyv/2J8r/9efK//Xnyv/158r/9efK//Xnyv/158r/9ee
K//Xnyv/154r/9eeK//Xnyv/154r/9efK//Xnyv/2J8r/9efK//Ynyv/2J8r/9eeK//Ynyv/2J8rggAA
AAAAAAAAAAAAAN+fNRjYoCz02J8r/9ifK//Ynyv/2J8r/9efK//Ynyv/158r/9efK//Ynyz/158r/9if
LP/Ynyz/2KAs/9ifLP/YoCz/2KAs/+bBd//58eD/+fDf/9uoPv/Ynyz/2J8r/9ifLP/Ynyz/2J8r/9if
LP/YoCz03581GAAAAAAAAAAA2qIsltmgLP/ZoCz/2aAs/9mgLP/ZoCz/2KAs/9mgLP/YoCz/2KAs/9mg
LP/YoCz/2aAs/9mgLP/ZoCz/2aAs/92rRv/048L/////////////////6cqJ/9mgLP/ZoCz/2aAs/9mg
LP/YoCz/2aAs/9igLP/aoiyWAAAAAP+qVQbboi3x2qEt/9mgLP/aoS3/2aAs/9mgLP/aoS3/2aAs/9qh
Lf/aoS3/2qEt/9qhLf/aoS3/2qEt/9mhLv/qyon//fr1///////////////////////y3rf/2aAs/9qh
Lf/ZoCz/2aAs/9qhLf/ZoCz/2qEt/9uiLfH/qlUG26IvR9qhLf/aoS3/2qEt/9qhLf/aoS3/2qEt/9uh
Lf/aoS3/26Et/9uhLf/aoS3/26Et/9qhLf/fr0z/9+vT//////////////////////////////////nx
3//aoS3/26Iu/9qhLf/aoS3/2qEt/9qhLf/aoS3/2qEt/9uiL0fboy6V26Iu/9uiLv/coi7/26Iu/9yi
Lv/coi7/26Iu/9yiLv/boi7/26Iu/9uiLv/boi7/26Iu//v27P//////////////////////////////
/////////v79/9ylNv/boi3/26Iu/9uiLv/boS3/26Iu/9uhLf/boS3/26MuldyjL7/coy7/3KMu/9yi
Lv/coy7/3KIu/9yiLv/coi7/3KIu/9yjMP/cpDP/26Iu/9yiLv/boi7/7M6S////////////////////
////////////////////////47Za/9yiLv/coy//3KMv/9yjLv/coy//3KMu/9yjLv/coy+/3qMv2d2j
Lv/doy7/3aMv/92jLv/doy//47NT/+3Rlv/36tH//vz5//79/P/04bz/4bBM/92jL//doy//7MyM////
///////////////////////////////////qxn7/3KMu/92jL//doy//3KMu/92jL//coy7/3KMu/96j
L9nepTDz3qQv/96kL//doy//3qQv/+jCdv/////////////////////////////////9+/f/7c6Q/96l
Mv/epDD/6sd///7+/f////////////////////////////DXpP/epC//3qQw/96kMP/epC//3qQw/96k
L//epC//3qQv8t6lMPPepDD/3qQw/9+lMP/epDD/36Yz/+vIgf/579v/////////////////////////
////////+O3X/+S2WP/fpTD/6cJ1//79+///////////////////////9ubG/96kMP/epC//3qQv/96k
MP/epC//3qQw/96kMP/gpjDz36Yw2eClMf/gpTH/36Uw/+ClMf/fpTD/36Uw/9+lMP/mumL/9OC4//79
+v///////////////////////v38//DWov/gqDj/6MBw//79+v/////////////////79ej/36Ux/+Cl
Mf/gpTH/36Uw/+ClMf/fpTD/36Uw/9+mMNngpjC/4KYx/+CmMf/gpjH/4KYx/+CmMf/gpjH/4KYx/+Cm
Mf/gpjH/4q5E/+7QlP/79Of///////////////////////v05v/pwHD/892z////////////////////
///hqTv/4KUw/+ClMP/gpjH/4KUw/+CmMf/gpjH/4qcxv+KmM5XhpjH/4aYx/+GmMf/hpjH/4aYx/+Gm
Mf/hpjH/4aYx/+GmMf/hpjH/4acy/+GnM//qwXD/9ubF//7+/f//////////////////////////////
/////////////+e4W//hpjL/4aYy/+GmMf/hpjL/4aYx/+GmMf/ipjKV4qkyR+GmMv/hpjL/4qcy/+Gm
Mv/ipzL/4qcy/+GmMf/ipzL/4aYx/+GmMf/ipzL/4aYx/+KnMv/ipzL/5LFM//HVnf/89uv/////////
////////////////////////68V4/+KnMv/ipzL/4qcy/+KnMv/ipzL/4qcy/+KpMkf/qlUG46g08eOo
M//ipzL/46gz/+KnMv/ipzL/46gz/+KnMv/jqDP/46gz/+KnMv/jqDP/4qcy/+KnMv/ipzL/4qcy/+Ko
Nf/rwnP/9uXD//79/P/////////////////v0JH/4qcy/+KnMv/jqDP/4qcy/+OoM//jqDTx/6pVBgAA
AADkqDOW46gz/+OoM//jqDP/46gz/+OoM//jqDP/46gz/+OoM//jqDP/5Kgz/+OoM//kqDP/5Kgz/+Sp
M//kqDP/5Kkz/+SpM//jqDP/5rFH/+/PkP/47NP//vv3/+zFd//jqDP/46gz/+OoM//jqDP/46gz/+So
M5YAAAAAAAAAAOqqNRjlqTT05ak0/+SpNP/lqTT/5ak0/+SpNP/lqTT/5Kk0/+SpNP/kqTP/5Kk0/+Sp
M//kqTP/5Kgz/+SpM//kqDP/5Kgz/+SpNP/kqDP/5Kk0/+SpNP/lqTT/5Kk0/+WpNP/lqTT/5Kgz/+Wp
NP/lqDP06qo1GAAAAAAAAAAAAAAAAOarNYLlqTT/5ak0/+WpNP/lqTT/5Kk0/+WpNP/kqTT/5Kk0/+Wp
NP/kqTT/5ak0/+WpNP/lqjT/5ak0/+WqNP/lqjT/5ak0/+WqNP/lqTT/5ak0/+WpNP/lqTT/5ak0/+Wp
NP/lqTT/5ak0/+arNYIAAAAAAAAAAAAAAAAAAAAA/7ZJB+eqNsfmqjX/5qo1/+aqNf/lqjT/5qo1/+Wq
NP/lqjT/5qo0/+WqNP/mqjT/5qo0/+aqNf/mqjT/5qo1/+aqNf/mqjT/5qo1/+aqNP/mqjT/5qo1/+aq
NP/mqjX/5qo1/+WqNP/nqjbH/7ZJBwAAAAAAAAAAAAAAAAAAAAAAAAAA6qo1GOesN+PmqjX/5qo1/+er
Nf/mqjX/56s1/+erNf/nqzb/56s1/+erNv/nqzb/5qo1/+erNv/mqjX/5qo1/+erNf/mqjX/56s1/+er
Nf/mqjX/56s1/+aqNf/mqjX/56w24+qqNRgAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA6aw3Lues
N+Pnqzb/6Ks2/+erNv/oqzb/6Ks2/+erNf/oqzb/56s1/+erNf/nqzb/56s1/+erNv/nqzb/6Kw2/+er
Nv/orDb/6Kw2/+erNv/orDb/56s2/+esN+PosDktAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAA6rVAGOmtN8forDb/6aw3/+isNv/orDb/6Kw2/+isNv/orDb/6Kw2/+isNv/orDb/6Kw2/+is
Nv/orDb/6Kw2/+isNv/orDb/6Kw2/+isNv/prTfH6rVAGAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAA/7ZJB+mtN4LprDf06aw2/+msNv/orDb/6aw2/+isNv/orDb/6a03/+is
Nv/prTf/6a03/+msN//prTf/6aw3/+msN//prTf06a05gv+2SQcAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAPS1QBjrrziW66848eqtOP/qrjj/6q04/+qt
OP/qrTf/6q04/+qtN//qrTf/6a03/+qtN//qrjfx6604lvS1QBgAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAD/1VUG7bA5R+yv
OJXsrzi/66842euuOfPrrjnz66842eyvOL/srziV7bA5R//VVQYAAAAAAAAAAAAAAAAAAAAAAAAAAAAA
AAAAAAAAAAAAAAAAAAAAAAAA/8AD//8AAP/8AAA/+AAAH/AAAA/gAAAHwAAAA8AAAAOAAAABgAAAAQAA
AAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAAIAAAAGAAAABwAAAA8AA
AAPgAAAH8AAAD/gAAB/8AAA//wAA///AA/8="""


NOT_GBR_ERRORS = {
    'SubTasksError', 'SubtasksMixinException', 'TestFailures', 'BaseConsoleMixinException', 'ReleaseSubTasksMixinException', 'MetrikaDictsUploadRestartException',
}

DEVTOOLS_ERRORS = {
    'ArcCommandFailed',
}


def is_gbr_error(exc_type):
    return exc_type.__name__ not in NOT_GBR_ERRORS


def is_devtools_error(exc_type):
    return exc_type.__name__ in DEVTOOLS_ERRORS


def _cls_order(cls, inspect_cls):
    if cls == inspect_cls:
        return 2
    elif not issubclass(cls, binary_task.LastBinaryTaskRelease):
        return 1
    else:
        return 0


def _call_with_parents(cls, function, exclude_parents):
    """
    :param type cls:
    :param str function:
    :param Set[type] exclude_parents:
    :return:
    """

    bases = sorted(
        enumerate(inspect.getmro(cls)),
        key=lambda idx_parent_cls: (_cls_order(idx_parent_cls[1], inspect_cls=cls), idx_parent_cls[0])
    )
    functions = [
        (base.__name__, base.__dict__[function])
        for _, base in bases
        if (
            function in base.__dict__ and
            base.__dict__[function].__name__ != 'wrapper' and  # don't call wrappers when inherit from class with @with_parents
            base.__name__ not in exclude_parents
        )
    ]

    def wrapper(self, *args, **kwargs):
        for cls, function in functions:
            logging.debug('Calling {} of cls {}'.format(function.__name__, cls))
            function(self, *args, **kwargs)

    setattr(cls, function, wrapper)


def exc_handler_deco(f):
    def exc_handler(self, *args, **kwargs):
        try:
            f(self, *args, **kwargs)
        except Exception:
            exc_type, exc_val, exc_tb = sys.exc_info()
            tb = traceback.extract_tb(exc_tb)
            frame = tb[-1]
            logging.debug('Last frame %s', frame)
            if six.PY2:
                file = frame[0]
            else:
                file = frame.filename
            if '/metrika/' not in file and not file.startswith('metrika/'):
                if is_devtools_error(exc_type):
                    self.comment(
                        'Фатальная ошибка произошла где-то вне кода тасок, обратитесь в '
                        '<a href="https://forms.yandex-team.ru/surveys/65090/">'
                        '<img src="https://yastatic.net/s3/frontend/forms/v25.56.0/public/i/icons/color/favicon-16.png" width="15"/>&ThinSpace;Devtools</a>'
                    )
            else:
                if is_gbr_error(exc_type):
                    self.comment(
                        'Фатальная ошибка произошла где-то в коде тасок, обратитесь в '
                        '<a href="https://t.me/joinchat/Bvt2F0Ji2a_iwKDkMlrlFA"><img src="data:image/x-icon;base64,{}" width="15"/>&ThinSpace;ГБР</a>'.format(
                            TELEGRAM_IMG_BASE64  # bc of csp
                        )
                    )
            raise

    exc_handler.__name__ = six.ensure_str('{}_exc_handler'.format(f.__name__))
    return exc_handler


def with_parents(cls):
    """
    Декоратор класса для вызова всех `on_` методов вместе с методами родителей класса в порядке наследования:
    cls.mro + cls

    Поведение методов можно переопределить декоратором `exclude_parents`

    :param type cls:
    :return:
    """

    for function in WRAPPED_FUNCTIONS:
        self_function = cls.__dict__.get(function)
        exclude_parents = getattr(self_function, 'exclude_parents', set())
        if self_function and exclude_parents is True:
            continue

        _call_with_parents(cls, function, exclude_parents)

        setattr(cls, function, exc_handler_deco(cls.__dict__[function]))

    return cls


def _exclude_parents_deco(function, classes):
    """
    :param function function:
    :param classes:
    :return:
    """

    function.exclude_parents = [cls.__name__ for cls in classes]
    return function


def exclude_parents(*classes_or_function):
    """
    Декоратор для исключения вызовов методов родителей класса декорируемого метода.
    Если не указаны конкретные, то исключаются все

    Examples:
        @exclude_parents  -- исключаются все родители
        def on_execute(self):
            pass

        @exclude_parents(sdk2.Task)  -- исключается только `sdk2.Task`
        def on_execute(self):
            pass

    :param Union[List[type], function] classes_or_function: Список классов для исключения или декорируемый метод
    :return:
    """

    if len(classes_or_function) == 1 and inspect.isfunction(classes_or_function[0]):
        classes_or_function[0].exclude_parents = True
        return classes_or_function[0]
    elif all(inspect.isclass(cls) for cls in classes_or_function):
        return partial(_exclude_parents_deco, classes=classes_or_function)
    else:
        raise ValueError('Value must be list of classes or function')


class BaseMetrikaMixin(
    LastRefreshableBinary,
    AutoTicketMixin,
    SubtasksMixin
):
    @property
    def real_author(self):
        """
        :param sdk2.Task self:
        """
        if config.Registry().common.installation == misc.Installation.LOCAL:
            return config.Registry().client.sdk.svn.arcadia.rw.user
        else:
            return self.author

    @property
    def link(self):
        """
        :param sdk2.Task self:
        """
        return get_task_link(self.id)

    def comment(self, info):
        """
        :param sdk2.Task self:
        """
        self.set_info(info, do_escape=False)

    @staticmethod
    def get_secret(uuid, key):
        return sdk2.yav.Secret(uuid).data()[key]

    @memoize
    def wd_path(self, *path):
        """
        :param sdk2.Task self:
        """
        return self.path('wd', *path)

    @memoize
    def wd(self, *path):
        return self.wd_path(*path).as_posix()


class BaseMetrikaTask(BaseMetrikaMixin, sdk2.Task):
    class Requirements(MetrikaTinyRequirements):
        pass


class BaseMetrikaLargeTask(BaseMetrikaMixin, sdk2.Task):
    class Requirements(MetrikaLargeRequirements):
        pass
