# coding=utf-8
import functools
import logging
import re

from sandbox.common.utils import progressive_waiter
from sandbox.projects.common.conductor import Api, OauthAuth
from sandbox.projects.metrika.utils import settings
from sandbox.sdk2 import Vault

logger = logging.getLogger("conductor")


class Conductor:

    @staticmethod
    def comment_generator(task, st_ticket=None):
        comment = "Создан в Sandbox: {}".format(task.link)

        release_issue_key = st_ticket.get("release_issue_key")
        if release_issue_key:
            comment += "\nРелизный тикет: http://st/{}".format(release_issue_key)

        issue_key = st_ticket.get("issue_key")
        if issue_key:
            comment += "\nТикет: http://st/{}".format(issue_key)

        description = st_ticket.get("description")
        if description:
            comment += "\n\n{}".format(description)

        return comment

    def __init__(self, task):
        self.__api = Api(task=task, auth=OauthAuth(Vault.data(settings.owner, settings.conductor_token)))
        self.__api.timeout = 30

    def create_conductor_ticket(self, packages, issue_key=None, branch="testing", no_autoinstall=False, projects=None, release_issue_key=None, description=None, deploy_groups=None):
        resp = self.__api.ticket_add(
            packages=packages,
            branch=branch,
            no_autoinstall=no_autoinstall,
            projects=projects,
            deploy_groups=deploy_groups,
            comment_generator=self.comment_generator,
            st_ticket={
                "release_issue_key": release_issue_key,
                "issue_key": issue_key,
                "description": description
            }
        )

        match = re.search('^URL: https://c.yandex-team.ru/tickets/(?P<ticket_id>[0-9]+)$', resp, flags=re.MULTILINE)
        if match:
            ticket_id = match.group("ticket_id")
            logger.info("Conductor ticket {} created.".format(ticket_id))
            return ticket_id
        else:
            raise Exception("Получен некорректный ответ от Conductor: {}".format(resp))

    def wait_conductor_tickets(self, conductor_tickets):
        initial_tick = 10
        max_tick = 100
        max_wait = 1000

        if not progressive_waiter(initial_tick, max_tick, max_wait, functools.partial(self.__check_all_tickets_success, conductor_tickets=conductor_tickets))[0]:
            raise Exception("Не все тикеты в Conductor успешно завершились")

    def __check_all_tickets_success(self, conductor_tickets):
        return all(self.__check_ticket_success(ticket_id) for ticket_id in conductor_tickets)

    def __check_ticket_success(self, ticket_id):
        conductor_temporary_statuses = ["need_info", "failed", "maybe_missing", "frozen"]
        conductor_fail_statuses = ["obsolete", "revoked"]

        status = self.__api.ticket_status(ticket_id)

        if status == "done":
            return True

        if status in conductor_temporary_statuses:
            raise Exception("Conductor тикет {} во временном статусе {}".format(ticket_id, status))

        if status in conductor_fail_statuses:
            raise Exception("Conductor тикет {} в терминальном статусе {}".format(ticket_id, status))

        logger.info("Conductor тикет {} в статусе {}".format(ticket_id, status))

        return False

    def __get_package_info(self, package_name):
        conductor = self.__api
        with conductor.session as session:
            r = session.get(
                conductor.build_url('/api/v2/packages/' + package_name),
                headers=conductor.auth.headers, cookies=conductor.auth.cookies, timeout=conductor.timeout
            )
            if r.status_code == 404:
                return {}

            r.raise_for_status()
            return r.json()['data']

    def package_is_deleted(self, package_name):
        info = self.__get_package_info(package_name)
        return not bool(info) or info['attributes']['deleted']
