# coding=utf-8
from __future__ import unicode_literals

import logging
import os
import tempfile

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.sdk2.helpers import gdb as gdb_helpers


class BaseConsoleMixinException(TaskError):
    pass


class BaseConsoleMixin(object):
    """
    Mixin для выполнения консольных команд
    """

    def _execute_shell(self, args, cwd=None, env=None, timeout=None, verbose=True, logger="console"):
        """
        :param sdk2.Task self:
        :param args: command-line arguments, single string or sequence
        :param cwd: current dir
        :param env: environment variables
        :return: exit code
        """
        if not isinstance(args, list):
            args = args.split()

        with sdk2.helpers.ProcessLog(
            self, logger=logger,
            stdout_level=logging.INFO, stderr_level=logging.ERROR
        ) as pl:
            process = sdk2.helpers.subprocess.Popen(args, cwd=cwd, env=env, stdout=pl.stdout, stderr=sdk2.helpers.subprocess.STDOUT)
            exit_code = process.wait(timeout=timeout)

            if verbose:
                pl.logger.info("=" * 50)
                pl.logger.info("Execute shell")
                pl.logger.info("\t\tCMD: %s", str(args))
                if len(args) > 1 and args[0] == '/bin/bash' and os.path.exists(args[1]):
                    pl.logger.info("\t\tSCRIPT: %s", open(args[1]).read())
                pl.logger.info("\t\tCWD: %s", cwd)
                pl.logger.info("\t\tENV: %s", env)
                pl.logger.info("\t\tTIMEOUT: %s", timeout)
                pl.logger.info("")
                pl.logger.info("\t\tEXIT CODE: %s", exit_code)

            if exit_code:
                self.set_info(gdb_helpers.get_html_view_for_logs_file("stdout", pl.stdout.path.relative_to(self.log_path()), self.log_resource), do_escape=False)

            return exit_code

    def _execute_shell_and_check(self, args, cwd=None, env=None, timeout=None, verbose=True, logger="console"):
        """
        if exit code is not zero throws exception
        :param args: command-line arguments, single string or sequence
        :param cwd: current dir
        :param env: environment variables
        :return: nothing
        """
        exitcode = self._execute_shell(args, cwd=cwd, env=env, timeout=timeout, verbose=verbose, logger=logger)

        if exitcode:
            raise BaseConsoleMixinException("Command\n{!r}\nFAILED. Exit code: {!r}".format(args, exitcode))

    def _execute_script(self, script, cwd=None, env=None, timeout=None, logger="console"):
        with tempfile.NamedTemporaryFile() as script_file:
            script_file.write(script)
            script_file.flush()
            self._execute_shell_and_check(['/bin/bash', script_file.name], cwd=cwd, env=env, timeout=timeout, logger=logger)
