# -*- coding: utf-8 -*-

import logging

from sandbox.common.config import Registry
from sandbox.common.types.misc import Installation
from sandbox.common.types.task import Status
from sandbox.projects.common.juggler import jclient


class JugglerReporterMixin(object):
    """
    Данный mix-in предназначен для Sandbox-задачи, которая представляет собой мониторируемую сущность,
    статус отсылаемого события определяется на основе статуса задачи.
    """
    juggler_host = 'sandbox'
    juggler_service = None

    def get_event(self):
        """
        :return: 4-tuple события - (service, host, status, description)
        """
        return None, None, None, None

    def _send_juggler_event(self, task_status):
        service, host, status, description = self.get_event()
        host = self.juggler_host or host
        service = self.juggler_service or service or self.type.name

        if not status:
            if task_status in Status.Group.SUCCEED:
                if self._juggler_warn_predicate():
                    status = 'WARN'
                else:
                    status = 'OK'
            elif task_status in Status.Group.SCHEDULER_FAILURE:
                status = 'CRIT'

        juggler_event = {
            'host': host,
            'service': service,
            'status': status,
            'description': '{} [ {} ]'.format(description or '', self.link).strip()
        }

        event_string = '{host}:{service} [{status}] {description}'.format(**juggler_event)

        if Registry().common.installation != Installation.LOCAL:
            if self._juggler_predicate(task_status):
                jclient.send_events_to_juggler(**juggler_event)
                logging.info('Juggler monitoring sent: {}'.format(event_string))
            else:
                logging.info("Don't send monitoring because of predicate: {}".format(event_string))

        else:
            logging.info("Don't send monitoring because of local installation: {}".format(event_string))

    def _juggler_predicate(self, status):
        return bool(self.scheduler)

    def _juggler_warn_predicate(self):
        return False

    def on_finish(self, previous_status, status):
        self._send_juggler_event(status)

    def on_break(self, previous_status, status):
        self._send_juggler_event(status)


class JugglerMultiReporterMixin(object):
    """
    Данный mix-in предназначен для Sandbox-задач, которые являются более чем одной мониторируемой сущностью,
    например, она обрабатывает несколько сущностей за раз и нужно отправлять события по каждой сущности.
    Статусы отсылаемых событий определяются на основе статуса задачи.
    """
    juggler_host = 'sandbox'
    juggler_service = None

    def get_events(self):
        """
        :return: список 4-tuple'ов - событий [(service, host, status, description), ...]
        """
        return [(None, None, None, None)]

    def _send_juggler_events(self, task_status):
        events = []
        event_strings = []
        for service, host, status, description in self.get_events():
            host = self.juggler_host or host
            service = self.juggler_service or service or self.type.name

            if not status:
                if task_status in Status.Group.SUCCEED:
                    if self._juggler_warn_predicate():
                        status = 'WARN'
                    else:
                        status = 'OK'
                elif task_status in Status.Group.SCHEDULER_FAILURE:
                    status = 'CRIT'

            description = '{} [ {} ]'.format(description or '', self.link).strip()

            # см. https://a.yandex-team.ru/arc_vcs/sandbox/projects/common/juggler/jclient.py?rev=r4306160#L13
            # host, service, status, description
            events.append((host, service, status, description))
            event_strings.append('{host}:{service} [{status}] {description}'.format(host=host, service=service, status=status, description=description))

        events_string = '\n'.join(event_strings)

        if Registry().common.installation != Installation.LOCAL:
            if self._juggler_predicate(task_status):
                jclient.send_multi_events_to_jugger(events)
                logging.info('Juggler monitoring sent: {}'.format(events_string))
            else:
                logging.info("Don't send monitoring because of predicate: {}".format(events_string))

        else:
            logging.info("Don't send monitoring because of local installation: {}".format(events_string))

    def _juggler_predicate(self, status):
        return bool(self.scheduler)

    def _juggler_warn_predicate(self):
        return False

    def on_finish(self, previous_status, status):
        self._send_juggler_events(status)

    def on_break(self, previous_status, status):
        self._send_juggler_events(status)
