# coding=utf8
from __future__ import unicode_literals

import logging
from time import sleep

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types.task import Status
from sandbox.projects.common import utils as spcu
from sandbox.projects.metrika.utils.mixins.subtasks import SubTasksError
from sandbox.projects.metrika.utils.parameters import TrackerIssue


def set_skynet_id_attr(**kwargs):
    resources = sdk2.Resource.find(**kwargs).limit(0)
    for resource in resources:
        if resource.skynet_id is not None and resource.type.name != 'BUILD_LOGS':
            spcu.set_resource_attributes(resource.id, {"rbtorrent": resource.skynet_id})


# Пустой ресурс, необходимый для релиза тасок, не имеющих своих ресурсов
class MetrikaEmptyResource(sdk2.Resource):
    releasable = True
    releasers = ["METRIKA"]


class ReleasableTaskMixin(object):
    """
    Mixin, позволяющий осуществлять релиз таски, не имеющей своих ресурсов
    """

    def on_finish(self, prev_status, status):
        open(str(MetrikaEmptyResource(self, "Пустой ресурс Метрики", "empty_file").path), "a").close()


class ReleaseSubTasksMixinException(TaskError):
    pass


class ReleaseSubTasksMixin(object):
    """
    Mixin, позволяющий осуществлять релиз дочерних тасок YaPackage
    """

    TIME_TO_RELEASE = 5 * 60
    SLEEP_TIME = 15

    @property
    def releasable_tasks(self):
        """
        :param sdk2.Task self:
        """

        # query tasks, bc status refreshes in loop
        return sdk2.Task.find(type=['METRIKA_YA_PACKAGE', 'METRIKA_YA_MAKE'], status=Status.Group.SUCCEED, parent=self).limit(0)

    def on_release(self, parameters):
        """
        :param sdk2.Task self:
        """

        tracker_issue_parameters = [
            parameter.name
            for parameter in type(self).Parameters
            if getattr(type(self).Parameters, parameter.name).__name__ == TrackerIssue.__name__
        ]
        if tracker_issue_parameters:
            tracker_issue = getattr(self.Parameters, tracker_issue_parameters[0])
        else:
            tracker_issue = self.Context.release_issue_key

        for subtask in self.releasable_tasks:
            version = subtask.Context.output_resource_version or 'New version'
            if ":" in version:
                version = version.split(":")[1]
            subject = "{} by {}".format(version, subtask.author)
            if tracker_issue:
                issue_summary = self.st_client.issues[tracker_issue].summary
                subject = "{} {}\n{}".format(tracker_issue, issue_summary, subject)
            if parameters.get("release_subject"):
                subject = "{}\n{}".format(subject, parameters.get("release_subject"))
            release_params = {
                "task_id": subtask.id,
                "type": parameters["release_status"],
                "subject": subject,
                "message": parameters.get("release_comments"),
                "to": parameters.get("email_notifications", {}).get("to", []),
                "cc": parameters.get("email_notifications", {}).get("cc", []),
                "releaser": parameters.get("releaser", self.author)
            }
            self.server.release(release_params)

        wait_time = self.TIME_TO_RELEASE
        while wait_time > 0:
            waiting_subtasks = [t for t in self.releasable_tasks if t.status not in [Status.RELEASED, Status.NOT_RELEASED]]
            if waiting_subtasks:
                logging.debug('Waiting %s. Sleep %s', waiting_subtasks, self.SLEEP_TIME)
                sleep(self.SLEEP_TIME)
                wait_time -= self.SLEEP_TIME
            else:
                break
        else:
            raise ReleaseSubTasksMixinException('Timeout on children release')

        failed_tasks = []
        for subtask in self.releasable_tasks:
            if subtask.status == Status.NOT_RELEASED:
                failed_tasks.append(subtask)
            else:
                set_skynet_id_attr(task_id=subtask.id)
        if failed_tasks:
            raise SubTasksError(failed_tasks)


class ReleasableMixin(ReleasableTaskMixin, ReleaseSubTasksMixin):
    pass
