# coding=utf-8
from __future__ import unicode_literals

import hashlib
import logging

import six
from six.moves import collections_abc

from sandbox import common, sdk2
from sandbox.common.errors import TaskError, TaskFailure, UnknownTaskType
from sandbox.common.types import task as task_type
from sandbox.common.urls import get_task_link
from sandbox.projects.common import binary_task


class SubtasksException(Exception):
    def __init__(self, subtasks):
        if not isinstance(subtasks, collections_abc.Iterable):
            subtasks = [subtasks]
        self.tasks = [t if isinstance(t, sdk2.Task) else sdk2.Task[t] for t in subtasks]

    def get_task_info(self):
        return "Не все сабтаски завершились успешно:<br/>" + "<br/>".join(['<a href="{link}"># {id}</a> {description}'.format(
            link=get_task_link(task.id),
            id=task.id,
            description=task.Parameters.description
        ) for task in self.tasks])


class SubTasksFailure(SubtasksException, TaskFailure):
    pass


class SubTasksError(SubtasksException, TaskError):
    pass


class SubtasksMixinException(TaskError):
    pass


class SubtasksMixin(object):
    """
    Миксин для запуска пачки сабтасок, ожидания и проверки.
    Принимает на вход список пар (таска, параметры) subtasks=[(MyTask, {param1: val1})], где MyTask или тип таски или ее класс.
    НЕ НУЖНО оборачивать мемойзом, внутри уже реализована логика с ним. Название мемойза можно либо передать явно через `stage_name`, либо оно посчитается само от параметра `subtasks`.
    По очереди запускает все задачи и кладет их айдишники в  `subtasks_variable`, если указана. Ждать сабтаски или нет определяет флаг `wait`.
    После ожидания, если хотя бы одна таска завершается неуспешно, то кидается исключение. После перезапуска родительской таски запускаются ТОЛЬКО НЕУСПЕШНЫЕ сабтаски.
    Успешность определяется методом `check_subtask`, принимающим на вход объект сабтаски.
    Если нужно выполнить какие-нибудь действия до/после запуска каждой задачи, их можно передать в виде функций в `after/before_subtask_enqueued`.
    Аналогично с действиями после запуска/ожидания всех задач `after_subtasks_enqueued/waited`.
    Если родительская таска запускается из архива, то его можно прокинуть в сабтаски указав флаг `pass_tasks_resource`.

    Примеры:
    1. subtasks_ids = self.run_subtasks(subtasks=[(MyTask, {param1: val1})])
    2. self.run_subtasks(subtasks=[(MyTask, {param1: val1})], subtasks_variable=self.Context.subtasks)
    3. self.run_subtasks(subtasks=[(MyTask, {param1: val1}), (MyTask, {param1: val2})], wait=False, after_subtasks_enqueued=my_cool_hook)
    """

    def check_subtask(self, subtask):
        return (subtask.status in task_type.Status.Group.SUCCEED or 'OK' in subtask.Parameters.tags) and 'FAIL' not in subtask.Parameters.tags

    def after_subtask_enqueued(self, subtask):
        pass

    def before_subtask_enqueued(self, subtask):
        pass

    def after_subtasks_enqueued(self):
        pass

    def after_subtasks_waited(self):
        pass

    def if_subtasks_failed(self, failed_subtasks):
        pass

    def _to_str(self, e):
        if isinstance(e, dict):
            return six.text_type(sorted((self._to_str(k), self._to_str(v)) for k, v in e.items()))
        elif isinstance(e, collections_abc.Iterable) and not isinstance(e, six.string_types + (type, sdk2.Resource)):  # type из-за того что внезапно sdk2.Task Iterable
            return six.text_type(sorted(self._to_str(x) for x in e))
        else:
            return six.text_type(e)

    def _canonize(self, tasks):
        if not isinstance(tasks, list):
            tasks = [tasks]

        for i, item in enumerate(tasks):
            if len(item) == 2:  # task, params
                tasks[i] = list(item) + [{}]  # add requirements
            params = tasks[i][1]
            for k in list(params.keys()):
                if not isinstance(k, six.string_types):
                    params[k.name] = params.pop(k)

        return tasks

    def run_subtasks(
        self, subtasks, subtasks_variable=None, stage_name=None, pass_tasks_resource=True, run=True, wait=True, info=True,
        before_subtask_enqueued=None, after_subtask_enqueued=None,
        after_subtasks_enqueued=None, after_subtasks_waited=None,
        if_subtasks_failed=None,
    ):
        """
        :param sdk2.Task self:
        :param List[Tuple[Task|str, dict, dict*]] subtasks: Список пар (Класс таски или название, параметры) или троек (Класс таски или название, параметры, требования)
        :param list subtasks_variable: Список, в который будут добавляться айди задач
        :param str stage_name: Название мемойза
        :param bool pass_tasks_resource: Передать сабтаскам архив задач
        :param bool wait: Флаг запуска задач
        :param bool wait: Флаг ожидания задач
        :param bool info: Печатать или нет информацию о запущенных сабтасках
        :param before_subtask_enqueued: Тригер, вызываемый перед запуском задачи
        :param after_subtask_enqueued: Тригер, вызываемый после запуска задачи
        :param after_subtasks_enqueued: Тригер, вызываемый после запуска всех задач
        :param after_subtasks_waited: Тригер, вызываемый после ожидания всех задач
        :param if_subtasks_failed: Тригер, вызываемый если какие-то из задач завершились неуспешно
        :return: List[int]: Список айди задач
        :raise SubTasksError: Одна из задач не прошла проверку `check_subtask` после выполнения
        """

        subtasks = self._canonize(subtasks)

        if not run:
            wait = False

        if not stage_name:
            stage_str = self._to_str(subtasks)
            logging.debug('Subtasks stage: %s', stage_str)
            stage_name = hashlib.md5(stage_str).hexdigest()

        if not self.Context._run_count:
            self.Context._run_count = {}

        if stage_name not in self.Context._run_count:
            self.Context._run_count[stage_name] = 1

        if subtasks_variable is None:
            if not self.Context._subtasks:
                self.Context._subtasks = {}
            if stage_name not in self.Context._subtasks:
                self.Context._subtasks[stage_name] = []
            subtasks_variable = self.Context._subtasks[stage_name]
        elif subtasks_variable and len(subtasks) != len(subtasks_variable):
            logging.debug('The number of subtasks has changed')
            subtasks_variable[:] = []

        with self.memoize_stage['run_subtasks_{}_{}'.format(stage_name, self.Context._run_count[stage_name])](commit_on_entrance=False):
            if not subtasks_variable:
                subtasks_variable.extend(None for _ in range(len(subtasks)))

            tasks = {
                False: [],  # not enqueued
                True: []  # enqueued
            }
            for i, (subtask_id, (subtask_type, params, requirements)) in enumerate(zip(subtasks_variable, subtasks)):
                if subtask_id and self.check_subtask(sdk2.Task[subtask_id]):
                    continue

                if not isinstance(subtask_type, six.string_types):
                    subtask_type = subtask_type.type  # sdk1 tasks fix

                try:
                    subtask = sdk2.Task[subtask_type](self, **params)

                    if pass_tasks_resource:
                        # пробрасываем только если указан кастом и тестинг, потому что для стейбла проброс вызывает боль при релизе дочерних тасок
                        if (
                            isinstance(self, binary_task.LastBinaryTaskRelease) and
                            self.Parameters.binary_executor_release_type in {'custom', 'testing'} and
                            isinstance(subtask, binary_task.LastBinaryTaskRelease)
                        ):
                            subtask.Parameters.binary_executor_release_type = 'custom'
                        subtask.Requirements.tasks_resource = self.Requirements.tasks_resource
                        subtask.save()

                    if requirements:
                        for key, value in requirements.items():
                            setattr(subtask.Requirements, key, value)
                        subtask.save()

                    subtasks_variable[i] = subtask.id
                except UnknownTaskType:
                    raise
                except Exception:
                    logging.exception('Can\'t create and save task %s with params %s', subtask_type, params)
                    tasks[False].append(subtask_type)
                    continue

                (before_subtask_enqueued if before_subtask_enqueued else self.before_subtask_enqueued)(subtask)

                if not run:
                    tasks[True].append(subtask.id)
                    continue

                try:
                    subtask.enqueue()
                except common.errors.TaskNotEnqueued:
                    logging.exception('Can\'t enqueue task %s with params %s', subtask_type, params)
                    tasks[False].append(subtask.id)
                    continue
                else:
                    tasks[True].append(subtask.id)
                    (after_subtask_enqueued if after_subtask_enqueued else self.after_subtask_enqueued)(sdk2.Task[subtask.id])

            (after_subtasks_enqueued if after_subtasks_enqueued else self.after_subtasks_enqueued)()

            for is_enqueued, subtasks in tasks.items():
                if subtasks:
                    if info or not is_enqueued:
                        self.set_info(
                            '<b><font color="{}">{}</font></b>:\n'.format(
                                'green' if is_enqueued else 'red',
                                'ENQUEUED' if is_enqueued else 'NOT ENQUEUED (see debug log)',
                            ) +
                            '\n'.join(
                                '<a href="{}">{} #{}</a> {}'.format(
                                    get_task_link(task_id), sdk2.Task[task_id].type.name, task_id, sdk2.Task[task_id].Parameters.description
                                )
                                if isinstance(task_id, int) else
                                task_id  # тип таски, если нет id для `NOT ENQUEUED`
                                for task_id in subtasks
                            ),
                            do_escape=False
                        )

            if wait:
                if tasks[True]:
                    raise sdk2.WaitTask(tasks[True], task_type.Status.Group.FINISH + task_type.Status.Group.BREAK)
                elif tasks[False]:
                    raise SubtasksMixinException('All tasks are failed to enqueue')

        if wait:
            (after_subtasks_waited if after_subtasks_waited else self.after_subtasks_waited)()
            if any(task_id is None for task_id in subtasks_variable):
                self.Context._run_count[stage_name] += 1
                raise SubtasksMixinException('Some tasks have not been enqueued')

            failed_subtasks = [task_id for task_id in subtasks_variable if not self.check_subtask(sdk2.Task[task_id])]
            if failed_subtasks:
                self.Context._run_count[stage_name] += 1
                (if_subtasks_failed if if_subtasks_failed else self.if_subtasks_failed)(failed_subtasks)
                raise SubTasksError(failed_subtasks)

        return subtasks_variable
