# coding=utf-8
import logging

from sandbox.common.types.notification import Charset, Transport, Type, View
from sandbox.projects.common.link_builder import HREF_TO_ITEM
from sandbox.projects.common.telegram_api import notify
from sandbox.projects.metrika.utils import settings

TELEGRAM_RECIPIENTS = {
    "Metrika Jenkins Notifications": -1001100204684,
    "MetriQA Notifications": -1001119505188,
    "Metrika Java": -1001117922610,
    "Metrika QA": -1001524933995,
    "Release Approves": -1001086985122,

    "egatov": 61022087,
    "firstsvet": 282762140,
    "miptgirl": 70908039,
    "okunev": 119029501,
    "orantius": 131248693,
    "pavel-belikov": 409478737,
    "sourx": 158146317,
    "z-sergey": 157210881,
    "zgmnkv": 273289893,
}


class Notifications(object):

    @staticmethod
    def telegram(task, recipients, message):
        if not isinstance(recipients, (list, tuple, set)):
            recipients = [recipients]

        bot = notify.create_bot(task, settings.telegram_token, settings.owner)
        if bot is None:
            logging.error("Unable to get telegram bot. Skip telegram notifications.")
            return

        if recipients:
            logging.info("Telegram recipients: {}".format(recipients))
            logging.info("Telegram message: {}".format(message))

            chat_ids = set(TELEGRAM_RECIPIENTS[chat] if chat in TELEGRAM_RECIPIENTS
                           else TELEGRAM_RECIPIENTS["Metrika Jenkins Notifications"] for chat in recipients)
            notify.notify_chats(task, bot, chat_ids, Notifications.__append_message(task, message, "\n"))

    @staticmethod
    def email(task, recipients, subject, body):
        if not isinstance(recipients, (list, tuple, set)):
            recipients = [recipients]

        mail_tail = "@yandex-team.ru"
        recipients = [person if person.endswith(mail_tail) else person + mail_tail for person in recipients]

        logging.info("E-mail recipients: {}".format(recipients))
        logging.info("E-mail message: {}".format(body))

        try:
            task.server.notification(
                recipients=recipients,
                subject=subject,
                body=Notifications.__append_message(task, body),
                transport=Transport.EMAIL,
                type=Type.HTML,
                charset=Charset.UTF,
                task_id=task.id,
                view=View.DEFAULT
            )
        except Exception:
            logging.error("Unable to send email. Skip email notifications.", exc_info=True)

    @staticmethod
    def __append_message(task, message, line_delimiter="<br/>"):
        return "{message}{delimiter}{delimiter}{tail}".format(
            delimiter=line_delimiter,
            message=message.strip(),
            tail="Задача в Sandbox: {task_link}{delimiter}Автор задачи: {author_link}".format(
                delimiter=line_delimiter,
                task_link=HREF_TO_ITEM.format(
                    link=task.link,
                    name="#" + str(task.id)
                ),
                author_link=HREF_TO_ITEM.format(
                    link="https://staff.yandex-team.ru/{}".format(task.real_author),
                    name=task.real_author + "@"
                ),
            ).strip()
        )
