# coding=utf-8
from __future__ import unicode_literals

import re

import requests
import six
from requests.packages.urllib3.util.retry import Retry
from sandbox.projects.sandbox.resources import LXC_CONTAINER

from sandbox import common, sdk2
from sandbox.common import utils as common_utils
from sandbox.projects.common.build.parameters import ArcadiaPatch
from sandbox.sdk2 import Parameters
from sandbox.sdk2.parameters import ArcadiaUrl, Group, RadioGroup, String


def choices(choices, autocomplete=False):
    """
    Создает и дополняет список выборов до 7 элементов, чтобы в интерфейсе отображалось с автодополнением и можно было удалять значение после выбора

    :param List[str|Tuple[str, str]] choices: Список выборов или пар выборов
    :return:
    """

    if isinstance(choices, list) and not isinstance(choices[0], tuple):
        choices = [(c, c) for c in choices]
    if autocomplete:
        choices += [('', '')] * (7 - len(choices))

    return choices


class RegexpString(sdk2.parameters.String):
    # https://a.yandex-team.ru/review/1147795/details
    regexp = None

    @classmethod
    def cast(cls, value):
        if cls.regexp is None:
            raise ValueError('regexp argument required')

        if value is not None:
            value = six.text_type(value)
            if not re.match(cls.regexp, value):
                raise ValueError("'{0}' doesn't match '{1}'".format(value, getattr(cls.regexp, 'pattern', cls.regexp)))

        return value


class TrackerIssue(RegexpString):
    autolink = True
    description = 'Тикет'
    regexp = r'[A-Z]+-\d+$'


class VcsParameters(Parameters):
    with Group("VCS") as vcs_common_block:
        with RadioGroup('VCS') as vcs:
            vcs.values.arcadia = vcs.Value(value='Arcadia', default=True)

            with vcs.value['arcadia']:
                arcadia_url = ArcadiaUrl("Arcadia URL", required=True, default="arcadia-arc:/")
                arcadia_patch = ArcadiaPatch(
                    "Патч", required=False, default="",
                    description="Рекомендуется накладывать патч на полный SVN клон репозитория. Нельзя использовать селективный чекаут."
                )
                arcadia_path = String("Путь внутри Аркадии до проекта", required=True)


class ArcadiaURL(sdk2.parameters.String, sdk2.parameters.ArcadiaUrl):
    description = 'Arcadia URL'
    default_value = 'arcadia-arc:/#trunk'

    choices = choices([
        'arcadia-arc:/#trunk', 'arcadia-arc:/#users/{author}/{ticket}',
        'arcadia:/arc/trunk/arcadia', 'arcadia:/arc/trunk/arcadia@revision'
    ], autocomplete=True)

    @classmethod
    def cast(cls, value):
        return sdk2.parameters.ArcadiaUrl.cast(value)


def hide(cls):
    """
    Прячем параметры под чекбокс

    Либо
    class Parameters(sdk2.Parameters):
        class SecretParameters:  !!! без наследования от sdk2.Parameters
            kek = String('kek')

        _secrets = hide(SecretParameters)  # обязательно присвоить в какую-нибудь переменную (название не имеет значения)

    Либо
    class SecretParameters(sdk2.Parameters):
        kek = String('kek')

    class Parameters(sdk2.Parameters):
        _secrets = hide(SecretParameters)
    """

    class Params(sdk2.Parameters):
        with sdk2.parameters.Group('Task development parameters (don\'t change)') as secret_admin_params:
            show_secret_admin_params = sdk2.parameters.Bool('Show task development parameters', default=False, description='Yes, I know what i\'m doing')
            with show_secret_admin_params.value[True]:
                if issubclass(cls, sdk2.Parameters):
                    _ = cls()
                else:
                    _ = type(cls.__name__, (sdk2.Parameters, cls), cls.__dict__)()

    return Params()


class LastCosmosContainerResource(sdk2.parameters.Container):
    """
    Параметр - ресурс, у которого дефолтное значение - последний ресурс Cosmos контейнера
    """

    resource_type = LXC_CONTAINER.name

    @common_utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            owner="METRIKA",
            attrs={"purpose": "cosmos"},
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class LastFrontendTestsContainerResource(sdk2.parameters.Container):
    """
    Параметр - ресурс, у которого дефолтное значение - последний ресурс контейнера с окружением для тестов фронта
    """

    resource_type = LXC_CONTAINER.name

    @common_utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            owner="METRIKA",
            attrs={"purpose": "frontend-tests"},
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class LastKnightContainerResource(sdk2.parameters.Container):
    """
    Параметр - ресурс, у которого дефолтное значение - последний ресурс контейнера с окружением knight-trusty
    """

    resource_type = LXC_CONTAINER.name

    @common_utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            owner="METRIKA",
            attrs={"metriqa_jenkins_slave": "knight-trusty"},
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class LastPeasantContainerResource(sdk2.parameters.Container):
    """
    Параметр - ресурс, у которого дефолтное значение - последний ресурс контейнера с окружением Peasant
    """

    resource_type = LXC_CONTAINER.name

    @common_utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            owner="METRIKA",
            attrs={"metriqa_jenkins_slave": "peasant"},
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class LastMySqlContainerResource(sdk2.parameters.Container):
    """
    Параметр - ресурс, у которого дефолтное значение - последний ресурс контейнера с клиентскими программами mysql
    """

    resource_type = LXC_CONTAINER.name

    @common_utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            owner="METRIKA",
            attrs={"mysql": "5.7"},
            limit=1,
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None


class PackageVersion(sdk2.parameters.String):
    """
    Параметр - номер версии, у которого дефолтное значение - версия пакета в проде на заданной группе хостов
    """
    MT_API_URL = "https://mtapi.mtrs.yandex-team.ru/v1/packages/"
    TEMPLATE_URL = MT_API_URL + "{}/{}/{}"
    host_group = None  # conductor
    package = None
    env = None  # deploy

    @common.utils.classproperty
    def default_value(cls):
        if cls.package is not None:
            s = requests.session()
            s.mount(cls.MT_API_URL, requests.adapters.HTTPAdapter(max_retries=Retry(total=20, backoff_factor=1)))
            if cls.host_group is not None:
                url = cls.TEMPLATE_URL.format("group", cls.host_group, cls.package)
            elif cls.env is not None:
                url = cls.TEMPLATE_URL.format("deploy", cls.package, cls.env)
            else:
                return

            r = s.get(url).json()
            if r.get("result"):
                return r["data"]


class DataCenterParameters(Parameters):
    with sdk2.parameters.RadioGroup("ДЦ", description="Дата-центр для MaaS и CaaS") as data_center:
        data_center.values["sas"] = data_center.Value(default=True)
        data_center.values["vla"] = None
