# coding=utf-8
import itertools
import json
import logging
import sys
import traceback
from datetime import datetime

from sandbox import common
from sandbox import sdk2
from sandbox.common.config import Registry
from sandbox.common.types import task
from sandbox.common.types.misc import Installation
from sandbox.common.urls import get_task_link
from sandbox.projects.common.decorators import memoized_property
from sandbox.projects.common.juggler import jclient
from sandbox.projects.metrika import utils
from sandbox.projects.metrika.utils import settings, custom_report_logger
from sandbox.projects.metrika.utils.pipeline.common import DTF
from sandbox.projects.metrika.utils.pipeline.pipeline_errors import PipelineInternalError, PipelineFailure, \
    PipelineError, PipelineAbortError
from sandbox.projects.metrika.utils.pipeline.pipeline_state import StateManager
from sandbox.projects.metrika.utils.pipeline.pipeline_view import StateViewModel
from sandbox.projects.metrika.utils.pipeline.retry_memoize_creator import MemoizeCreator
from sandbox.projects.metrika.utils.base_metrika_task import BaseMetrikaTask

logger = logging.getLogger('pipeline')

# https://yc.yandex-team.ru/folders/foori5uktoh2v12cbltq/managed-clickhouse/cluster/b0750f6d-ae39-4c46-b3e7-021e6bad2182
STAT_HOSTS = ['man-17bbhfsco4ib01nq.db.yandex.net', 'sas-x49hcnhq49jky587.db.yandex.net', 'vla-dg2pin1wjpqu03a6.db.yandex.net']
STAT_DB = 'sandbox_stat'

AUDIT_REQ_TEMPLATE = "INSERT INTO {database}.{table} (EventTime, author, owner, task_id, task_type, event, tags, params) FORMAT JSONEachRow"
STAT_REQ_TEMPLATE = "INSERT INTO {database}.{table} (task_id, StageName, StageTitle, StartTs, FinishTs, RetryState) FORMAT JSONEachRow"


class Audit(object):
    # экземпляр конвейера создан - особой ценности не имеет
    CREATED = 'CREATED'

    # конвейер запущен - событие начала конвейера
    STARTED = 'STARTED'

    # промежуточное событие - начало стадии
    STAGE_STARTED = 'STAGE_STARTED'
    # промежуточное собыние - успешное завершение стадии
    STAGE_FINISHED = 'STAGE_FINISHED'

    # конвейер приостановлен из-за поломок в нём
    BROKEN = 'BROKEN'
    # конвейер приостановлен из-за поломок вне его
    EXCEPTION = 'EXCEPTION'

    # конвейер продолжен после одного из двух предыдущих событий
    RESUMED = 'RESUMED'

    # конвейер завершён успешно с позитивным результатом
    FAILURE = 'FAILURE'
    # конвейер завершён успешно с негативным результатом
    SUCCESS = 'SUCCESS'

    # конвейере прерван, результата достигнуто не будет
    ABORT = 'ABORT'


class PipelineBase(object):
    class Context(sdk2.Task.Context):
        state = {}

    def create_stages(self):
        """
        Должен быть перегружен в наследнике
        :return: iterable, каждый элемент которого представляет собой стадию, стадией может быть:
        имя функции или функция в классе задачи, либо iterable, первый элемент которого
        то, что указано выше, а второй - title стадии
        """
        return []

    def on_execute(self):
        # тут строим пайплайн, по ходу валидируя, что есть все нужные функции у всех стадий, а стадии только что созданы и не завершены
        if not self.Context.state:
            self.Context.state = StateManager().init(self.create_stages()).validate(self).state

        if not self.Context.state['stages']:
            return

        # проверяем, выполнялся ли конвейер
        # если да, то отправляем сообщение о запуске конвейера в БД
        # аудит - STARTED
        if not self.state.started:
            self.state.start()
            self.Context.save()
            self._audit(Audit.STARTED)

        # тут выполняем пайплайн
        # к этому моменту self.Context.state - провалидирован
        # получаем очередную стадию, узнаём, финальная ли она,
        # выполняем тем или иным образом, обновляем self.Context.state, идём далее (или выбрасываем исключение)
        while not self.state.is_completed:
            stage = self.state.next_stage
            # проверить до старта - сколько ретраев в стадии, если 0 - то начинаем новую - не шлём аудит
            # если более нуля и у последнего есть finish_ts, то это повтор после прерывания
            if stage.retry_count > 0 and stage.retries_wrapped[-1].finish_ts is not None:
                self._audit(Audit.RESUMED)
            stage.start(self.log_resource.id)
            self.Context.save()
            try:
                getattr(self, stage.func_name)()
            except (common.errors.Wait, common.errors.TemporaryError):
                logger.info("Sandbox flow control exception. Reraise", exc_info=sys.exc_info())
                raise
            except PipelineInternalError:
                stage.finish(False)
                self.Context.save()
                logger.error("Internal pipeline error. Please fix pipeline.", exc_info=sys.exc_info())
                # аудит - BROKEN
                self._audit(Audit.BROKEN)
                raise
            except PipelineError:
                stage.finish(False)
                self.Context.save()
                logger.warning("Retrieable pipeline error.", exc_info=sys.exc_info())
                # аудит - EXCEPTION
                self._audit(Audit.EXCEPTION)
                raise
            except PipelineFailure as e:
                if self.state.is_on_final_stage:
                    # пайплайн дошёл до конца с провалом, ретраев более не будет
                    stage.finish(True)
                    self.Context.save()
                    logger.info("Pipeline failed. Sorry.", exc_info=sys.exc_info())
                    raise
                else:
                    # это ошибка в логике пайплайна
                    stage.finish(False)
                    self.Context.save()
                    logger.warning("Pipeline failure not in final stage. Please fix pipeline.", exc_info=sys.exc_info())
                    # аудит - BROKEN
                    self._audit(Audit.BROKEN)
                    raise PipelineInternalError("Ошибка в логике пайплайна - это не финальная стадия", e)
            except PipelineAbortError:
                stage.finish(False)
                logger.info("Pipeline is not able to achieve any result.")
                raise
            except Exception as e:
                stage.finish(False)
                self.Context.save()
                logger.info("Other non special exception. Reraise as PipelineError.", exc_info=sys.exc_info())
                # аудит - EXCEPTION
                self._audit(Audit.EXCEPTION)
                self.set_info(PipelineError("Обнаружена общая ошибка.", e).get_task_info())
                raise

            # Стадия дошла до конца, успешно.
            stage.finish(True)
            self.Context.save()

    def on_finish(self, prev_status, status):
        if not self.Context.state['stages']:
            return

        # аудит - SUCCESS or FAILURE в зависимости от status
        if status == task.Status.SUCCESS:
            self._audit(Audit.SUCCESS)
        elif status == task.Status.FAILURE:
            if not self.state.is_completed:
                self._audit(Audit.ABORT)
            else:
                self._audit(Audit.FAILURE)
        else:
            logger.warning("Unsupported final status: {}".format(status))

        # отсылка статистики по стадиям
        self._stat(self.state)

    def on_before_timeout(self, seconds):
        self.Context.save()

    def timeout_checkpoints(self):
        return [0]

    @property
    def memoize_stage(self):
        """
        Изменяем поведение memoize_stage так, что бы оно работало в пределах одной повторной попытки
        :return:
        """
        return MemoizeCreator(self, self.state.current_stage.func_name, self.state.current_stage.retry_count)

    @property
    def memoize_stage_global(self):
        return super(PipelineBase, self).memoize_stage

    @property
    def state(self):
        return StateManager(self.Context.state)

    @property
    def view(self):
        return StateViewModel(self)

    @memoized_property
    def clickhouse(self):
        from metrika.pylib.clickhouse import ClickHouse
        return ClickHouse(
            host=STAT_HOSTS, port=8443, user='stat', password=sdk2.Vault.data('METRIKA', 'stat-mdb-password'), round_robin_hosts=False, https=True, verify=False
        )

    def _audit(self, event):
        from metrika.pylib.clickhouse import Query

        audit = {
            'EventTime': datetime.now().strftime(DTF),
            'author': self.author,
            'owner': self.owner,
            'task_id': self.id,
            'task_type': self.type.name,
            'event': event,
            'tags': [str(tag) for tag in list(self.Parameters.tags)],
            'params': ["{}={}".format(k, v) for k, v in dict(self.Parameters).iteritems()]
        }
        logger.info("Audit: {}".format(audit))
        if Registry().common.installation != Installation.LOCAL and self.owner == settings.owner:
            try:
                query = AUDIT_REQ_TEMPLATE.format(database=STAT_DB, table="sandbox_audit")
                self.clickhouse.execute(Query(query, method='POST', data=json.dumps(audit)))
                jclient.send_events_to_juggler("pipeline", "audit", "OK", get_task_link(self.id))
            except Exception:
                logger.warning("Fail to send audit data.", exc_info=True)
                jclient.send_events_to_juggler("pipeline", "audit", "CRIT", get_task_link(self.id) + " " + traceback.format_exc())

    def _stat(self, state):
        from metrika.pylib.clickhouse import Query

        # отправляем все строки из списка словарей
        stats = list(itertools.chain.from_iterable([[
            {
                'task_id': self.id,
                'StageName': stage.func_name,
                'StageTitle': stage.title,
                'StartTs': retry.start_ts,
                'FinishTs': retry.finish_ts,
                'RetryState': 'SUCCESS' if retry.is_completed else 'FAIL'
            } for retry in stage.retries_wrapped] for stage in state.stages_wrapped]))

        logger.info("Stat: {}".format(stats))
        if Registry().common.installation != Installation.LOCAL:
            try:
                query = STAT_REQ_TEMPLATE.format(database=STAT_DB, table="sandbox_statistics")
                self.clickhouse.execute(Query(query, method='POST', data="\n".join([json.dumps(stat) for stat in stats])))
                jclient.send_events_to_juggler("pipeline", "audit", "OK", get_task_link(self.id))
            except Exception:
                logger.warning("Fail to send statistics data.", exc_info=True)
                jclient.send_events_to_juggler("pipeline", "audit", "CRIT", get_task_link(self.id) + " " + traceback.format_exc())

    @sdk2.header()
    @custom_report_logger
    def header(self):
        if self.Context.state:
            return utils.render("view.html.jinja2", {"state": self.view})


class PipelineBaseTask(PipelineBase, BaseMetrikaTask):
    pass
