# coding=utf-8
from sandbox import common


class PipelineError(common.errors.TaskError):
    """
    Не финальное исключение, бросается из любого места, в него оборачиваются любые другие исключения, кроме
    TemporaryError и Wait

    https://julien.danjou.info/python-exceptions-guide/
    """

    def __init__(self, message, cause=None):
        super(PipelineError, self).__init__(message)
        self.cause = cause

    def format_cause(self):
        if self.cause:
            return self.cause.get_task_info() if hasattr(self.cause, 'get_task_info') else '{}: {}'.format(type(self.cause).__name__, self.cause)
        else:
            return ""

    def get_preambule(self):
        return "Ошибка в конвейере."

    def get_epilog(self):
        return "Выполнение может быть продолжено после устранения причин ошибки."

    def get_task_info(self):
        return "{preambule}\n{message}\n{cause}\n{epilog}".format(preambule=self.get_preambule(), message=str(self), cause=self.format_cause(), epilog=self.get_epilog())


class PipelineInternalError(PipelineError):
    """
    Особое исключение, которое бросается в случае ошибок в инфраструктуре пайплайна
    """

    def get_preambule(self):
        return "Внутренняя ошибка конвейера."


class PipelineAbortError(common.errors.TaskFailure):
    """
    Особое исключение, которое бросается в том случае, когда продолжить пайплайн невозможно
    """

    def get_task_info(self):
        return "Непреодолимая ошибка конвейера. Пайплайн прерван."


class PipelineFailure(common.errors.TaskFailure):
    """
    Финальное исключение, может быть брошено только из финальной стадии.
    Если не из финальной стадии - оборачивается в PipelineError
    """

    def get_task_info(self):
        return "Выполнение конвейера завершено с негативным результатом. Подробности - в логах."
