# coding=utf-8
import inspect
import logging
import pprint
from datetime import datetime

import six

from sandbox.projects.metrika import utils
from sandbox.projects.metrika.utils.pipeline.common import DTF
from sandbox.projects.metrika.utils.pipeline.pipeline_errors import PipelineInternalError

logger = logging.getLogger('pipeline-state')


class RetryManager(object):
    def __init__(self, storage=None):
        self.state = storage

    def init(self, start_ts, log_resource_id):
        if self.state is not None:
            raise PipelineInternalError("state in RetryManager is not empty: {}".format(self.state))
        else:
            self.state = {}

        self.state['start_ts'] = start_ts
        self.state['finish_ts'] = None
        self.state['is_completed'] = False
        self.state['logs'] = [log_resource_id]

        logger.info("Retry started on {}".format(start_ts))

        return self

    @property
    def start_ts(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @property
    def finish_ts(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @finish_ts.setter
    def finish_ts(self, value):
        self.state[inspect.currentframe().f_code.co_name] = value

    @property
    def is_completed(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @is_completed.setter
    def is_completed(self, value):
        self.state[inspect.currentframe().f_code.co_name] = value

    @property
    def logs(self):
        return self.state[inspect.currentframe().f_code.co_name]

    def finish(self, finish_ts, is_completed):
        if self.finish_ts:
            raise PipelineInternalError("state already finished: {}".format(self.state))

        self.finish_ts = finish_ts
        self.is_completed = is_completed

        logger.info("Retry finished on {} completed: {}".format(finish_ts, is_completed))


class StageManager(object):

    def __init__(self, storage=None):
        self.state = storage

    def init(self, func_name, title):
        if self.state is not None:
            raise PipelineInternalError("state in StageManager is not empty: {}".format(self.state))
        else:
            self.state = {}

        self.state['func_name'] = func_name
        self.state['title'] = utils.decode(title)
        self.state['retries'] = []

        logger.info("Stage initialized: {}".format(self.state))

        return self

    @property
    def func_name(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @property
    def title(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @property
    def retries(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @property
    def retries_wrapped(self):
        return [RetryManager(retry) for retry in self.retries]

    def start(self, log_resource_id):
        # надо проверить, завершился ли предыдущий ретрай или мы продолжаем после Wait-исключений
        if self.retries and self.retries_wrapped[-1].finish_ts is None:
            # продолжаем после Wait
            self.retries_wrapped[-1].logs.append(log_resource_id)
            logger.info("Stage continues in old retry...")
        else:
            # начинаем новую попытку
            self.retries.append(RetryManager().init(datetime.now().strftime(DTF), log_resource_id).state)

    def finish(self, is_completed):
        self.retries_wrapped[-1].finish(datetime.now().strftime(DTF), is_completed)

    @property
    def retry_count(self):
        return len(self.retries)

    @property
    def is_completed(self):
        return self.retries_wrapped[-1].is_completed if self.retry_count > 0 else False

    @property
    def start_ts(self):
        return self.retries_wrapped[0].start_ts if self.retry_count > 0 else None

    @property
    def finish_ts(self):
        return self.retries_wrapped[-1].finish_ts if self.retry_count > 0 else None


class StateManager(object):

    def __init__(self, storage=None):
        self.state = storage

    def init(self, stages):
        if self.state is not None:
            raise PipelineInternalError("state in StateManager is not empty: {}".format(self.state))
        else:
            self.state = {}

        # stages - iterable, каждый элемент либо имя-или-функция, либо iterable, в котором в первом
        # элементе имя-или-функция, а во втором - title
        def process_stage(s, t):
            if isinstance(s, six.string_types):
                func_name = s
            elif hasattr(s, 'im_func'):
                func_name = s.im_func.func_name
            else:
                raise PipelineInternalError("{} должно быть либо строкой, либо функцией".format(s))

            return StageManager().init(func_name, t if t else func_name).state

        self.state['stages'] = []
        if len(stages) > 0:
            for func, title in stages:
                self.stages.append(process_stage(func, title))
            self.state['started'] = False

        logger.info("State initialized: {}".format(self.state))

        return self

    @property
    def started(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @started.setter
    def started(self, value):
        self.state[inspect.currentframe().f_code.co_name] = value

    @property
    def stages(self):
        return self.state[inspect.currentframe().f_code.co_name]

    @property
    def stages_wrapped(self):
        return [StageManager(stage) for stage in self.stages]

    def validate(self, current_task):
        logger.debug("Stages:\n{}".format(pprint.pformat(self.stages)))
        orphans = filter(lambda stage: not StageManager(stage).is_completed and not hasattr(current_task, StageManager(
            stage).func_name), self.stages)
        if orphans:
            raise PipelineInternalError("Orphan stages: {}".format(
                ", ".join([pprint.pformat(orphan) for orphan in orphans])))

        return self

    def start(self):
        logger.info("Start pipeline")
        if self.started:
            raise PipelineInternalError("Конвейер уже был запущен")
        else:
            self.started = True

    @property
    def is_completed(self):
        return self.stages_wrapped[-1].is_completed

    @property
    def next_stage(self):
        return next(x for x in iter(self.stages_wrapped) if not x.is_completed)

    @property
    def current_stage(self):
        return next(reversed([x for x in self.stages_wrapped if x.retry_count > 0]))

    @property
    def is_on_final_stage(self):
        return self.current_stage.state is self.stages[-1]

    @property
    def start_ts(self):
        return self.stages_wrapped[0].start_ts if self.started and len(self.stages) > 0 else None

    @property
    def finish_ts(self):
        return self.stages_wrapped[-1].finish_ts if self.is_completed and len(self.stages) > 0 else None
