# coding=utf-8
import logging

from sandbox import sdk2
from sandbox.common import utils as common_utils
from sandbox.common.types import resource as common_resource, task as common_task
from sandbox.projects.metrika.utils import settings

METRIKA_RELEASERS = [settings.owner]


class BaseMetrikaTestsDataResource(sdk2.Resource):
    ttl = "inf"
    share = True
    restart_policy = common_resource.RestartPolicy.DELETE


class MetrikaCoreB2bTestsData(BaseMetrikaTestsDataResource):
    name = "METRIKA_CORE_B2B_TESTS_DATA"


class AppmetricaCoreB2bTestsData(BaseMetrikaTestsDataResource):
    name = "APPMETRICA_CORE_B2B_TESTS_DATA"


class ClickhouseTestsData(BaseMetrikaTestsDataResource):
    pass


class CaasClickhouseData(BaseMetrikaTestsDataResource):
    pass


class MetrikaGoodGoals(BaseMetrikaTestsDataResource):
    pass


class BaseMetrikaBinaryResource(sdk2.Resource):
    any_arch = False
    auto_backup = True
    executable = True
    releasable = True
    releasers = METRIKA_RELEASERS


class BaseMetrikaConfigResource(sdk2.Resource):
    any_arch = False
    auto_backup = True
    executable = True
    releasable = True
    releasers = METRIKA_RELEASERS


class MetrikaClickhouseBinary(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True
    releasers = METRIKA_RELEASERS
    version = sdk2.Attributes.String('Version', required=True)
    in_production = sdk2.Attributes.Bool('In production', default=False)
    restart_policy = common_resource.RestartPolicy.IGNORE


class MetrikaInformerDataGetterBinary(BaseMetrikaBinaryResource):
    relative_binary_path = sdk2.Attributes.String(
        "Relative path to actual binary within resource",
        default="informer-data-getter"
    )


class MetrikaAntifraudTablesUploaderBinary(BaseMetrikaBinaryResource):
    relative_binary_path = sdk2.Attributes.String(
        "Relative path to actual binary within resource",
        default="antifraud-tables-uploader"
    )


class MetrikaInformerData(sdk2.Resource):
    ttl = 2
    share = True
    restart_policy = common_resource.RestartPolicy.DELETE
    releasable = True
    releasers = [settings.owner]
    environment = sdk2.Attributes.String('environment')


class MetrikaBishopConfig(sdk2.Resource):
    share = False

    program = sdk2.Attributes.String('Program', required=True)
    environment = sdk2.Attributes.String('Environment', required=True)
    version = sdk2.Attributes.Integer('Version')

    def fetch(self, token):
        from metrika.pylib.bishop import Bishop

        bishop_client = Bishop(token=token)
        version, _, config = bishop_client.get_config(self.program, self.environment)
        self.version = version

        resource_data = sdk2.ResourceData(self)
        resource_data.path.write_text(config)
        resource_data.ready()

        logging.info('Config fetched: program={program}, environment={environment}, version={version}'.format(
            program=self.program,
            environment=self.environment,
            version=version
        ))

        return str(resource_data.path)


class MetrikaWatch(sdk2.Resource):
    any_arch = False
    auto_backup = True
    releasable = True


class LastReleasedWatchJSResource(sdk2.parameters.Resource):
    @common_utils.classproperty
    def default_value(self):
        items = sdk2.Task.server.resource.read(type=MetrikaWatch, attrs={"released": common_task.ReleaseStatus.STABLE}, limit=1)["items"]
        return items[0]["id"] if items else None


class MetrikaWatchJsRelease(sdk2.Resource):
    any_arch = True
    auto_backup = True
    releasable = True


class ClickHouseFunctionalTestsData(BaseMetrikaTestsDataResource):
    name = "CLICKHOUSE_FUNCTIONAL_TESTS_DATA"


class QEMUImageMetrika(sdk2.Resource):
    """
    Abstract Metrika QEMU image
    """

    auto_backup = True
    releasable = True
    releasers = METRIKA_RELEASERS


class QEMUImageMetrikaBaseTrusty(QEMUImageMetrika):
    """
    Metrika-Trusty based qemu images, built with BuildPortoLayer
    """
    name = "QEMU_IMAGE_METRIKA_BASE_TRUSTY"


class Phantom2dDebugLogs(sdk2.Resource):
    """
    MTRSADMIN-4150
    """
    name = "PHANTOM2D_DEBUG_LOGS"


class Phantom2dTestData(sdk2.Resource):
    """
    METRIQA-2870
    """
    name = "PHANTOM2D_TEST_DATA"
    auto_backup = True
    releasable = True
    releasers = METRIKA_RELEASERS


class MetrikaBstrBinary(sdk2.Resource):
    name = "METRIKA_BSTR_BINARY"
    releasable = True
    releasers = METRIKA_RELEASERS


class MetrikaJugglerChecksBundle(sdk2.Resource):
    ttl = "inf"


class MetrikaLoadTestingAmmofile(sdk2.Resource):
    name = "METRIKA_LOAD_TESTING_AMMOFILE"
    auto_backup = True
    releasable = True
    releasers = METRIKA_RELEASERS


class MetrikaIsRobot(sdk2.Resource):
    auto_backup = True
    releasable = True
    releasers = ["ANTIFRAUD"]


class DefaultValueResourceMixin(object):
    @common_utils.classproperty
    def default_value(cls):
        items = sdk2.Task.server.resource.read(
            type=cls.resource_type,
            attrs=cls.attrs,
            state=common_resource.State.READY,
            limit=1
        )["items"]
        if items:
            return items[0]["id"]
        else:
            return None
