import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.helpers import process
from sandbox.projects.mobile_apps.utils.resources import AndroidSdk
import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

_logger = logging.getLogger(__name__)

PLATFORM_VERSION = '28'  # android-28
BUILD_TOOLS_VERSION = '29.0.2'
INSTALL_SDK_SCRIPT = '''
#!/bin/bash -e
wget https://dl.google.com/android/repository/commandlinetools-linux-6858069_latest.zip
unzip commandlinetools-linux-6858069_latest.zip
mkdir -p android-sdk/cmdline-tools
mv cmdline-tools android-sdk/cmdline-tools/tools

export ANDROID_SDK_ROOT=`pwd`/android-sdk

yes | $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --licenses
for platform_ver in {}; do
  echo 'exec sdkmanager --install "platforms;android-$platform_ver"'
  $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "platforms;android-$platform_ver"
done

for tool_ver in {}; do
  echo 'exec sdkmanager --install "build-tools;$tool_ver"'
  $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "build-tools;$tool_ver"
done

$ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --uninstall "emulator"

ndk_version_list="{}"
if [ $ndk_version_list != "None" ]; then
  for ndk_ver in $ndk_version_list; do
    echo 'exec sdkmanager --install "ndk;$ndk_ver"'
    $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "ndk;$ndk_ver"
  done;
fi

install_emulator="{}"
if [ $install_emulator == "True" ]; then
 echo 'exec sdkmanager --install "emulator"'
 $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "emulator"
fi

install_packages="{}"
package_filename="{}"
if [ $install_packages == "True" ]; then
 echo 'exec sdkmanager --package_file=$package_filename'
 $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --package_file=$package_filename
fi

rm -rf $ANDROID_SDK_ROOT/licenses/
'''


class AndroidSdkPreparerParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group('android_sdk parameters') as params:
        platform_version = sdk2.parameters.String(
            'Android SDK platform version',
            default=None, )
        build_tools_version = sdk2.parameters.String(
            'Android SDK build-tools version',
            default=None, )
        ndk_version = sdk2.parameters.String(
            'Android NDK version',
            default=None,
        )
        install_emulator = sdk2.parameters.Bool(
            'Install emulator',
            default=False,
        )
        packages = sdk2.parameters.List(
            'List of system images',
            default=None,
        )


class AndroidSdkPreparer(sdk2.Task):
    class Parameters(AndroidSdkPreparerParameters):
        pass

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        # wget+unzip+jdk8 https://sandbox.yandex-team.ru/resource/1682483488/view
        container_resource = 1682483488
        dns = ctm.DnsType.DNS64
        cores = 2
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    def on_save(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={'target': 'android_sdk_preparer/bin'}
        ).first().id

    def _execute_cmd(self, args, message):
        _logger.info('CMD: {}'.format(args))
        logger_name = 'android_sdk_preparing'
        cmd_logger = logging.getLogger(logger_name)
        cmd_process_log = sdk2.helpers.process.ProcessLog(logger=cmd_logger)
        rc = process.subprocess.Popen(
            args,
            shell=True,
            stdout=cmd_process_log.stdout,
            stderr=cmd_process_log.stderr
        ).wait()
        cmd_process_log.close()
        _logger.info('The command executed.')
        if rc:
            raise TaskFailure('{}{}'.format(message, rc))

    def _expand_version(self, version):
        if not version:
            return None
        expanded_version = []
        version_list = version.split(',')
        for each_element in version_list:
            if '-' in each_element:
                range_start, range_end = each_element.split('-')
                expanded_version += range(int(range_start), int(range_end) + 1)
            else:
                expanded_version.append(each_element)
        return ' '.join(str(element) for element in expanded_version)

    def _prepare_android_sdk(self):
        self._platform_version = self.Parameters.platform_version or PLATFORM_VERSION
        self._build_tools_version = self.Parameters.build_tools_version or BUILD_TOOLS_VERSION
        self._ndk_version = self.Parameters.ndk_version
        self._install_emulator = self.Parameters.install_emulator
        script_file = 'run.sh'
        packages_file = 'packages.list'
        self._expanded_platform_version = self._expand_version(self._platform_version)
        self._expanded_build_tools_version = self._expand_version(self._build_tools_version)
        self._expanded_ndk_version = self._expand_version(self._ndk_version)
        with open(script_file, 'w') as f:
            f.write(INSTALL_SDK_SCRIPT.format(self._expanded_platform_version, self._expanded_build_tools_version,
                                              self._expanded_ndk_version,
                                              self._install_emulator,
                                              len(self.Parameters.packages) > 0,
                                              packages_file))
        with open(packages_file, 'w') as f:
            f.write('\n'.join(self.Parameters.packages))

        self._execute_cmd(
            'bash {}'.format(script_file),
            'Can not install android-sdk. See common.log.')

    def _pack_android_sdk(self):
        android_sdk_path = './android-sdk'
        android_sdk_archive_path = 'android_sdk.tar'
        self._execute_cmd(
            'tar cf {} {}'.format(android_sdk_archive_path, android_sdk_path),
            'Can not pack android-sdk. See common.log.')

        version_string = 'sdk_{}+tools_{}'
        description_string = 'android sdk platform: {} + build_tools: {}'
        versions = [self._platform_version, self._build_tools_version]
        if self._ndk_version:
            versions.append(self._ndk_version)
            version_string = "{}+ndk_{{}}".format(version_string)
            description_string = "{} + ndk: {{}}".format(description_string)

        if self._install_emulator:
            version_string = "{}+emulator".format(version_string)
            description_string = '{} + emulator'.format(description_string)

        if self.Parameters.packages:
            versions.append(",".join(self.Parameters.packages))
            version_string = "{}+packages:{{}}".format(version_string)
            description_string = '{} + packages: {{}}'.format(description_string)

        attrs = {'version': version_string.format(*versions), 'platform': 'linux'}
        description = description_string.format(*versions)

        sdk2.ResourceData(AndroidSdk(self, description, android_sdk_archive_path, **attrs)).ready()

    def on_execute(self):
        with self.memoize_stage.prepare_android_sdk:
            self._prepare_android_sdk()
        self._pack_android_sdk()
