import logging

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.common.types import misc as ctm
from sandbox.common.types import resource as ctr
from sandbox.projects.common import binary_task
from sandbox.projects.mobile_apps.utils.resources import AndroidSdk
from sandbox.projects.mobile_apps.utils import shellexecuter

_logger = logging.getLogger(__name__)

INSTALL_SDK_SCRIPT = '''
#!/bin/bash -e
set -e
wget https://dl.google.com/android/repository/commandlinetools-linux-7302050_latest.zip
unzip commandlinetools-linux-7302050_latest.zip
mkdir -p android-sdk/cmdline-tools
mv cmdline-tools android-sdk/cmdline-tools/tools

export ANDROID_SDK_ROOT=`pwd`/android-sdk

yes | $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --licenses
for platform_ver in {}; do
  echo 'exec sdkmanager --install "platforms;android-$platform_ver"'
  $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "platforms;android-$platform_ver"
done

for tool_ver in {}; do
  echo 'exec sdkmanager --install "build-tools;$tool_ver"'
  $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "build-tools;$tool_ver"
done

ndk_version_list="{}"
if [ -n "$ndk_version_list" ]; then
  for ndk_ver in $ndk_version_list; do
    echo 'exec sdkmanager --install "ndk;$ndk_ver"'
    $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "ndk;$ndk_ver"
  done;
fi

system_images="{}"
if [ -n "$system_images" ]; then
  for image in $system_images; do
    echo 'exec sdkmanager --install "system_images;$image"'
    $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "system-images;$image"
  done;
fi

packages="{}"
if [ -n "$packages" ]; then
  for package in $packages; do
    echo 'exec sdkmanager --install "$package"'
    $ANDROID_SDK_ROOT/cmdline-tools/tools/bin/sdkmanager --install "$package"
  done;
fi

rm -rf $ANDROID_SDK_ROOT/licenses/
'''


class AndroidSdkPreparerParameters2(sdk2.Task.Parameters):
    """
    platforms(28,29,30)+build-tools(30.0.3)+system-images(android-28;google_apis;x86_64,android-30;google_apis;x86_64)+ndk(...)+packages(...)
    """

    with sdk2.parameters.Group('android_sdk parameters') as params:
        platform_version = sdk2.parameters.String(
            'Android SDK platform versions, space separated',
            required=True,
            description='Example: 28 29 31'
        )
        build_tools_version = sdk2.parameters.String(
            'Android SDK build-tools versions, space separated',
            required=True,
            description='Example: 29.0.3 30.0.3 '
        )
        ndk_version = sdk2.parameters.String(
            'Android NDK versions, space separated',
            default='',
            description='Example: 22.1.7171670 20.1.5948944'
        )
        system_images = sdk2.parameters.String(
            'List of system image paths, space separated, without prefix "system_images;"',
            default='',
            description='android-29;google_apis;x86 android-29;google_apis;x86_64'
        )
        packages = sdk2.parameters.String(
            'List of various packages paths, space separated',
            default='',
            description='extras;google;webdriver extras;android;m2repository'
        )

    ext_params = binary_task.binary_release_parameters(stable=True)


class AndroidSdkPreparer2(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Parameters(AndroidSdkPreparerParameters2):
        pass

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "ANDROID_SDK_PREPARER_2",
                      "released": self.Parameters.binary_executor_release_type,
                      "target": "android_sdk_preparer2/bin"},
            "state": [ctr.State.READY],
            "owner": "MOBDEVTOOLS",
        }

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        # wget+unzip+jdk8 https://sandbox.yandex-team.ru/resource/1682483488/view
        container_resource = 1682483488
        dns = ctm.DnsType.DNS64
        cores = 2
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass  # means that task do not use any shared caches

    def _concat_string(self, data):
        return ','.join(data.split(' '))

    def execute_cmd(self, args, message):
        _logger.info('CMD: {}'.format(args))
        shellexecuter.execute_on_platform(self, args, 'android_sdk_preparer2', False, message, cwd=None)

    def prepare_android_sdk(self):
        script_file = 'run.sh'
        with open(script_file, 'w') as f:
            f.write(INSTALL_SDK_SCRIPT.format(self.Parameters.platform_version,
                                              self.Parameters.build_tools_version,
                                              self.Parameters.ndk_version,
                                              self.Parameters.system_images,
                                              self.Parameters.packages))
        self.execute_cmd(
            'bash {}'.format(script_file),
            'Can not install android-sdk. See common.log.')

    def prepare_attributes(self):
        version_string = 'platforms({})+tools({})'
        description_string = 'android sdk platform: {} + build_tools: {}'
        versions = [self.platform_version, self.build_tools_version]
        if self.ndk_version:
            versions.append(self.ndk_version)
            version_string = "{}+ndk({{}})".format(version_string)
            description_string = "{} + ndk: {{}}".format(description_string)

        if self.system_images:
            versions.append(self.system_images)
            version_string = "{}+system-images({{}})".format(version_string)
            description_string = '{} + system-images: {{}}'.format(description_string)

        if self.packages:
            versions.append(self.packages)
            version_string = "{}+packages({{}})".format(version_string)
            description_string = '{} + packages: {{}}'.format(description_string)

        return version_string.format(*versions), description_string.format(*versions)

    def pack_android_sdk(self):
        android_sdk_path = './android-sdk'
        android_sdk_archive_path = 'android_sdk.tar'
        self.execute_cmd(
            'tar cf {} {}'.format(android_sdk_archive_path, android_sdk_path),
            'Can not pack android-sdk. See common.log.')

        version, description = self.prepare_attributes()
        ttl = 'inf' if self.Parameters.ext_params.binary_executor_release_type == 'stable' else '14'

        attrs = {'version': version,
                 'platform': 'linux',
                 'ttl': ttl,
                 'android_platform_versions': self.platform_version,
                 'build_tools': self.build_tools_version,
                 'ndk': self.ndk_version,
                 'system_images': self.system_images,
                 'packages': self.packages,
                 }
        sdk2.ResourceData(AndroidSdk(self, description, android_sdk_archive_path, **attrs)).ready()

    def on_execute(self):

        self.platform_version = self._concat_string(self.Parameters.platform_version)
        self.build_tools_version = self._concat_string(self.Parameters.build_tools_version)
        self.ndk_version = self._concat_string(self.Parameters.ndk_version)
        self.system_images = self._concat_string(self.Parameters.system_images)
        self.packages = self._concat_string(self.Parameters.packages)

        with self.memoize_stage.prepare_android_sdk:
            self.prepare_android_sdk()
        self.pack_android_sdk()
