import logging

from sandbox import sdk2
from sandbox.common.types import resource as ctr

from sandbox.projects.common import binary_task

_logger = logging.getLogger(__name__)


class BuildCounterCreatorParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group('generic_teamcity_runner parameters'):
        with sdk2.parameters.RadioGroup('Task type') as task_type:
            task_type.values['create_counter'] = task_type.Value(value='create counter', default=True)
            task_type.values['show_counter'] = task_type.Value(value='show counter')
            counter_name = sdk2.parameters.String(
                'counter name',
                default='test', )
            with task_type.value['create_counter']:
                initial_value = sdk2.parameters.Integer(
                    'initial value',
                    default=0, )

    ext_params = binary_task.binary_release_parameters(stable=True)


class BuildCounterCreator(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(BuildCounterCreatorParameters):
        pass

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "BUILD_COUNTER_CREATOR",
                      "released": self.Parameters.binary_executor_release_type,
                      "target": "build_counter_creator/bin"},
            "state": [ctr.State.READY],
            "owner": "MOBDEVTOOLS"
        }

    def _exists_counter(self):
        return self._client.exists(self._counter_path)

    def _create_client(self):
        import yt.wrapper as yt

        token = sdk2.Vault.data('robot-sbmonorepo', 'yt_token')
        yt_config = {
            'proxy': {'url': 'locke'},
            'token': token}
        self._client = yt.YtClient(config=yt_config)

        path = '//home/mobdevtools/build_counters'
        self._counter_name = self.Parameters.counter_name
        self._initial_value = self.Parameters.initial_value
        self._counter_path = '{}/{}'.format(path, self._counter_name)

    def _create_counter(self):
        if self._exists_counter():
            return 'Counter with name "{}" already exists'.format(self._counter_name)

        self._client.create('uint64_node', self._counter_path)
        self._client.set(self._counter_path, self._initial_value)

        return 'Counter with name "{}" successfully created with value {}.'.format(self._counter_name, self._initial_value)

    def _show_counter(self):
        if not self._exists_counter():
            return 'Counter with name "{}" does not exists'.format(self._counter_name)

        self._counter_value = self._client.get(self._counter_path)

        return 'The counter value of "{}" is {}.'.format(self._counter_name, self._counter_value)

    def on_execute(self):
        self._create_client()
        message = None
        if self.Parameters.task_type == 'create_counter':
            message = self._create_counter()
        else:
            message = self._show_counter()
        self.set_info(message, do_escape=False)
