import logging
import json
import textwrap

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.common.errors import TaskFailure
from sandbox.projects.common.vcs.arc import Arc

from sandbox.projects.common import binary_task


_logger = logging.getLogger(__name__)

HEAD = '<div style="font-size: 18; font-weight: 600; margin-left: 15px;">{head_name}</div>'
LINK = '<a href="{link}" style="margin-left: 30px;">{link}</a><br>'
REPORT = '<div style="font-size: 18; font-weight: 600;">Shifting finished with statistics: {} noted, {} successful shifted and {} failed. Detailed info:</div>{}'


class EternalBuildsShifterParameters(sdk2.Task.Parameters):
    ext_params = binary_task.binary_release_parameters(stable=True)

    with sdk2.parameters.Group('Eternal builds shifter parameters') as group:
        arc_token_name = sdk2.parameters.String(
            'Arc token name',
            required=True,
            default='ARC_TOKEN')
        with sdk2.parameters.RadioGroup('Parameters source') as parameters_source:
            parameters_source.values['sandbox_parameters'] = parameters_source.Value(value='Sandbox parameters', default=True)
            parameters_source.values['config'] = parameters_source.Value(value='Config')
            with parameters_source.value['sandbox_parameters']:
                yav_secret = sdk2.parameters.String(
                    'yav secret',
                    required=True, )
                with sdk2.parameters.RadioGroup('Shift by') as indicator:
                    indicator.values['project_ids'] = indicator.Value(value='Project ids', default=True)
                    indicator.values['build_ids'] = indicator.Value(value='Build ids')
                    with indicator.value['project_ids']:
                        included_teamcity_project_ids = sdk2.parameters.List(
                            'Included teamcity project ids', )
                        excluded_teamcity_project_ids = sdk2.parameters.List(
                            'Excluded teamcity project ids', )
                        included_build_type_ids = sdk2.parameters.List(
                            'Included build type ids', )
                        excluded_build_type_ids = sdk2.parameters.List(
                            'Excluded build type ids', )
                    with indicator.value['build_ids']:
                        build_ids = sdk2.parameters.List(
                            'Build ids',
                            value_type=sdk2.parameters.Integer, )
                yt_proxy = sdk2.parameters.String(
                    'Yt proxy',
                    default='hahn',
                    required=True, )
                yt_root = sdk2.parameters.String(
                    'Yt root',
                    required=True, )
                yt_token_key = sdk2.parameters.String(
                    'Yt token key',
                    required=True, )
                teamcity_url = sdk2.parameters.String(
                    'Teamcity url',
                    default='teamcity.yandex-team.ru',
                    required=True, )
                teamcity_token_key = sdk2.parameters.String(
                    'Teamcity token key',
                    required=True, )
                s3_bucket = sdk2.parameters.String(
                    'S3 bucket',
                    required=True, )
                s3_token_key_id = sdk2.parameters.String(
                    'S3 token key id',
                    required=True, )
                s3_token_key = sdk2.parameters.String(
                    'S3 token key',
                    required=True, )
                create_yt_structs = sdk2.parameters.Bool(
                    'Create yt structs',
                    default=False, )
            with parameters_source.value['config']:
                ssh_key = sdk2.parameters.Vault(
                    'SSH key', )
                config_path = sdk2.parameters.String(
                    'Config path', )


class EternalBuildsShifter(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Parameters(EternalBuildsShifterParameters):
        pass

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "ETERNAL_BUILDS_SHIFTER",
                      "released": self.Parameters.binary_executor_release_type,
                      "target": "eternal_builds_shifter/bin"},
            "state": [ctr.State.READY],
            "owner": "MOBDEVTOOLS",
        }

    @sdk2.header()
    def head(self):
        if self.Context.report_msg:
            return textwrap.dedent(self.Context.report_msg).strip()
        else:
            return 'Processing..'

    def _secret(self, secret_key):
        return sdk2.Vault.data(self.owner, '{}[{}]'.format(self._yav_secret, secret_key))

    def on_execute(self):
        _logger.info('Task started.')
        from devtools.mobile.eternal_builds.lib.shifter import Shifter

        if self.Parameters.parameters_source == 'config':
            arc = Arc(secret_name=self.Parameters.arc_token_name)
            ssh_key = self.Parameters.ssh_key
            with sdk2.ssh.Key(self, ssh_key.owner, ssh_key.name):
                repo = arc.mount_path(path=None, changeset=None, mount_point='arcadia', fetch_all=False)

            config = None
            with repo:
                with open(self.Parameters.config_path, 'r') as f:
                    config = json.load(f)
        else:
            config = {
                'yav_secret': self.Parameters.yav_secret,
                'teamcity_token': self.Parameters.teamcity_token_key,
                'yt_proxy': self.Parameters.yt_proxy,
                'yt_root': self.Parameters.yt_root,
                'yt_token': self.Parameters.yt_token_key,
                's3_bucket': self.Parameters.s3_bucket,
                's3_key_id': self.Parameters.s3_token_key_id,
                's3_key': self.Parameters.s3_token_key,
                'included_teamcity_project_ids': list(self.Parameters.included_teamcity_project_ids) if self.Parameters.indicator == 'project_ids' else None,
                'excluded_teamcity_project_ids': list(self.Parameters.excluded_teamcity_project_ids) if self.Parameters.indicator == 'project_ids' else None,
                'included_build_type_ids': list(self.Parameters.included_build_type_ids) if self.Parameters.indicator == 'project_ids' else None,
                'excluded_build_type_ids': list(self.Parameters.excluded_build_type_ids) if self.Parameters.indicator == 'project_ids' else None,
                'teamcity_url': self.Parameters.teamcity_url,
                'shift_requested_builds': (self.Parameters.indicator == 'build_ids'),
                'teamcity_build_ids': list(self.Parameters.build_ids) if self.Parameters.indicator == 'build_ids' else None,
                'create_yt_structs': self.Parameters.create_yt_structs
            }

        self._yav_secret = config['yav_secret']

        config['teamcity_token'] = self._secret(config['teamcity_token'])
        config['yt_token'] = self._secret(config['yt_token'])
        config['s3_key'] = self._secret(config['s3_key'])
        del config['yav_secret']
        shifter = Shifter(**config)

        succ_noted_count, succ_collected_count, failed_count, detailed_info = shifter.shift_builds()
        detailed_info = detailed_info.split('\n')
        detailed_info = [HEAD.format(head_name=row) if row.endswith(':') else LINK.format(link=row) for row in detailed_info]
        self.Context.report_msg = REPORT.format(succ_noted_count, succ_collected_count, failed_count, '\n'.join(detailed_info))
        if failed_count > 0:
            raise TaskFailure('Error during shifting.')
        _logger.info('Task finished.')
