import logging

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import notification as ctn
from sandbox.common.types import task as ctt
from sandbox.projects.common import binary_task
from sandbox.projects.mobile_apps.utils.resources import GenericResource
from sandbox.projects.mobile_apps.utils import shellexecuter

logger = logging.getLogger(__name__)


class GenericResourceBuilderParameters(sdk2.Task.Parameters):
    description = "Builds resource from user defined scenario"
    owner = "MOBDEVTOOLS"
    dump_disk_usage = False
    tags = ["mobdevtools", "GenericResourceBuilder"]

    notifications = [
        sdk2.Notification(
            statuses=[
                ctt.Status.FAILURE,
                ctt.Status.EXCEPTION,
                ctt.Status.TIMEOUT
            ],
            recipients=["mobdevtools-dev@yandex-team.ru"],
            transport=ctn.Transport.EMAIL
        )
    ]

    ext_params = binary_task.binary_release_parameters(stable=True)

    with sdk2.parameters.Group('Generic Resource Builder parameters') as gen_resource_builder_params:

        with sdk2.parameters.RadioGroup("Build resource on platform") as platform:
            platform.values["linux_x86"] = platform.Value("Linux")
            platform.values["macos_x86"] = platform.Value("Macos, Intel", default=True)
            platform.values["macos_arm"] = platform.Value("Macos, ARM")

        client_tag_os = sdk2.parameters.String(
            'OS',
            default=''
        )
        resource_name = sdk2.parameters.String(
            'Resource name',
            default='',
        )
        resource_version = sdk2.parameters.String(
            'Resource name',
            required=True,
        )
        resource_description = sdk2.parameters.String(
            'Describe resource',
            default='',
        )
        resource_folder = sdk2.parameters.String(
            'Folder which will be packed into resource',
            required=True,
        )
        build_script = sdk2.parameters.String(
            "Describe how to build resource",
            multiline=True,
            required=True,
        )
        runtime_script = sdk2.parameters.String(
            "This script will be embedded into resource and will be executed in runtime",
            multiline=True,
            default='',
        )
        ssh_key = sdk2.parameters.Vault(
            'SSH key', )


class GenericResourceBuilder(binary_task.LastBinaryTaskRelease, sdk2.Task):

    class Parameters(GenericResourceBuilderParameters):
        pass

    #  for 'macos_arm' choose binary 'bin(macos_x86)', because sandbox runs with Rosetta
    # 'macos_arm', 'macos_x86', 'linux_x86' are used as resource's platform and arch attributes
    platform_parameters = {"linux_x86": {"target": "generic_resource_builder/bin",
                                         "client_tags": "USER_MONOREPO & LINUX",
                                         },
                           "macos_x86": {"target": "generic_resource_builder/bin(macos_x86)",
                                         "client_tags": "USER_MONOREPO & MOBILE_MONOREPO",
                                         },
                           "macos_arm": {"target": "generic_resource_builder/bin(macos_x86)",
                                         "client_tags": "USER_MONOREPO & M1 & MOBILE_MONOREPO",
                                         },
                           }

    class Requirements(sdk2.Requirements):
        cores = 4
        ram = 8192

        class Caches(sdk2.Requirements.Caches):
            pass

    def on_execute(self):
        logger.info('Task started.')
        build(build_script=self.Parameters.build_script,
              runtime_script=self.Parameters.runtime_script,
              target_dir=str(self.path(self.Parameters.resource_folder)),
              task_dir=str(self.path())
              )
        self.create_resource()
        logger.info('Task finished.')

    def create_resource(self):
        attrs = {
            "platform": self.Parameters.platform,
            "resource_name": self.Parameters.resource_name,
            "resource_version": self.Parameters.resource_version,
            "description": self.Parameters.description,
        }
        resource = GenericResource(self,
                                   description=self.Parameters.resource_description,
                                   path=self.Parameters.resource_folder,
                                   **attrs
                                   )
        sdk2.ResourceData(resource).ready()
        logger.info('GenericResource with attributes {} was generated: {}'.format(attrs, resource.id))

    def _set_tags(self):
        self.Requirements.client_tags = self.platform_parameters[self.Parameters.platform]["client_tags"]
        if self.Parameters.client_tag_os != '':
            self.Requirements.client_tags = " & ".join((self.Requirements.client_tags, self.Parameters.client_tag_os))

    def on_create(self):
        self._set_tags()

    def on_save(self):
        self._set_tags()

        target = self.platform_parameters[self.Parameters.platform]["target"]
        binary_resource = sdk2.service_resources.SandboxTasksBinary.find(
            owner="MOBDEVTOOLS",
            attrs={
                "task_type": "GENERIC_RESOURCE_BUILDER",
                "target": target,
                "released": self.Parameters.binary_executor_release_type,
            }
        ).first()

        if not binary_resource:
            raise TaskFailure("Could not find binary task")
        self.Requirements.tasks_resource = binary_resource.id


def build(build_script, runtime_script, target_dir, task_dir):
    build_script_filename = "/".join([task_dir, "build.sh"])
    dump_script(build_script, build_script_filename)
    execute_cmd('bash -xe {}'.format(build_script_filename),
                'Failed to execute build script. See common.log.')
    dump_script(runtime_script, "/".join([task_dir, target_dir, "prepare.sh"]))


def dump_script(script, script_filename):
    with open(script_filename, 'w') as f:
        f.write(script)
    logger.info("Created {}".format(script_filename))


def execute_cmd(cmd, message):
    shellexecuter.execute_on_platform(sdk2.task.Task.current, cmd, 'generic_resource_builder', False, message, cwd=None)
