import logging
import requests
import sandbox.common.types.task as ctt

from sandbox import sdk2
from sandbox.sdk2.vcs.git import Git

_logger = logging.getLogger(__name__)

TASKS_BATCH = 15


class GitCacheHeaterParameters(sdk2.Task.Parameters):
    with sdk2.parameters.Group('git_cache_heater parameters'):
        with sdk2.parameters.RadioGroup('indicator') as indicator:
            indicator.values['tag'] = indicator.Value(value='tag', default=True)
            indicator.values['host'] = indicator.Value(value='host')
            with indicator.value['tag']:
                tag = sdk2.parameters.String(
                    'tag', )
        repo_url = sdk2.parameters.String(
            'Repo URL',
            default='ssh://git@bb.yandex-team.ru/mobile/monorepo.git', )
        branch = sdk2.parameters.String(
            'Git branch',
            default='refs/heads/master', )
        ssh_key = sdk2.parameters.Vault(
            'SSH key', )


class GitCacheHeater(sdk2.Task):
    class Parameters(GitCacheHeaterParameters):
        pass

    def on_save(self):
        self.Requirements.tasks_resource = sdk2.service_resources.SandboxTasksBinary.find(
            attrs={'target': 'git_cache_heater/bin'}
        ).first().id

    def _git_clone(self):
        git = Git(self.Parameters.repo_url, filter_branches=False)
        ssh_key = self.Parameters.ssh_key
        with sdk2.ssh.Key(self, ssh_key.owner, ssh_key.name):
            git.clone('.', self.Parameters.branch)

    def _enqued_subtasks(self, ids):
        params = {'indicator': 'host', 'repo_url': self.Parameters.repo_url, 'branch': self.Parameters.branch, 'ssh_key': self.Parameters.ssh_key, 'kill_timeout': 900}
        for id in ids[(self.Context.launch_step - 1) * TASKS_BATCH: min(self.Context.launch_step * TASKS_BATCH, len(ids) + 1)]:
            sub_task = GitCacheHeater(
                self,
                description='',
                **params
            )
            sub_task.Requirements.host = id
            sub_task.save()
            sub_task.enqueue()
            self.Context.running_tasks.append(sub_task.id)

    def _get_hosts(self):
        url = 'https://sandbox.yandex-team.ru/api/v1.0/client'
        headers = {'accept': 'application/json; charset=utf-8'}
        params = {'alive_offset': 0, 'tags': 'USER_MONOREPO', 'limit': 1000, 'offset': 0}
        request = requests.get(url, params=params, headers=headers)
        request.raise_for_status()
        response = request.json()
        ids = [item['id'] for item in response['items']]
        _logger.info(ids)
        return ids

    def on_execute(self):
        if self.Parameters.indicator == 'tag':
            with self.memoize_stage.prelaunch_preparation:
                self.Context.launch_step = 0
                self.Context.host_ids = self._get_hosts()

            self.Context.running_tasks = []

            self.Context.launch_step += 1
            with self.memoize_stage['enqued_{}'.format(self.Context.launch_step)]:
                self._enqued_subtasks(self.Context.host_ids)

            if len(self.Context.running_tasks) > 0:
                with self.memoize_stage['wait_{}'.format(self.Context.launch_step)]:
                    raise sdk2.WaitTask(self.Context.running_tasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)
        else:
            self._git_clone()
